#!/usr/bin/env python3
"""
Script de inicio para xfarma backend en producción.
Maneja puerto dinámico para deployment en Render.com
"""

import logging
import os

import uvicorn
from alembic import command
from alembic.config import Config

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


def apply_migrations():
    """Aplica migraciones de base de datos antes de iniciar el servidor."""
    try:
        logger.info("Aplicando migraciones de base de datos...")
        alembic_cfg = Config("alembic.ini")
        command.upgrade(alembic_cfg, "head")
        logger.info("✓ Migraciones aplicadas exitosamente")
        return True
    except Exception as e:
        logger.error(f"Error aplicando migraciones: {str(e)}")
        # En producción, continuamos aunque fallen las migraciones
        # para no interrumpir el servicio
        return False


if __name__ == "__main__":
    # Obtener puerto de variable de entorno (Render lo proporciona dinámicamente)
    port = int(os.getenv("PORT", 8000))

    # Obtener configuración del entorno
    environment = os.getenv("ENVIRONMENT", "development")

    # Aplicar migraciones antes de iniciar el servidor
    if environment == "production":
        apply_migrations()

    # Configurar uvicorn para producción o desarrollo
    if environment == "production":
        # Configuración para producción
        uvicorn.run(
            "app.main:app",
            host="0.0.0.0",
            port=port,
            workers=1,  # Render maneja escalado
            log_level="info",
            access_log=True,
            reload=False,
        )
    else:
        # Configuración para desarrollo local
        uvicorn.run("app.main:app", host="0.0.0.0", port=port, reload=True, log_level="debug")
