#!/usr/bin/env python3
"""
Script para verificar y aplicar migraciones automáticamente.
Se ejecuta en el startup de la aplicación en Render.
"""

import logging
import os
import sys

from alembic import command
from alembic.config import Config
from sqlalchemy import create_engine, text

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


def verify_and_apply_migrations():
    """
    Verifica el estado de las migraciones y las aplica si es necesario.
    """
    try:
        # Configurar Alembic
        alembic_cfg = Config("alembic.ini")

        # Intentar aplicar todas las migraciones pendientes
        logger.info("Verificando migraciones pendientes...")
        command.upgrade(alembic_cfg, "head")
        logger.info("✓ Migraciones aplicadas exitosamente")

        # Verificar que los cambios críticos estén aplicados
        database_url = os.getenv("DATABASE_URL")
        if database_url:
            engine = create_engine(database_url)
            with engine.connect() as conn:
                # Verificar que sale_date es TIMESTAMP
                result = conn.execute(
                    text(
                        """
                    SELECT data_type
                    FROM information_schema.columns
                    WHERE table_name = 'sales_data'
                    AND column_name = 'sale_date'
                """
                    )
                )
                row = result.fetchone()
                if row:
                    if "timestamp" in row[0].lower():
                        logger.info("✓ sale_date es TIMESTAMP")
                    else:
                        logger.warning(f"⚠ sale_date es {row[0]}, debería ser TIMESTAMP")

                # Verificar que las columnas obsoletas no existen
                obsolete_columns = ["ean_code", "therapeutic_category", "veterinary_use"]
                for col in obsolete_columns:
                    result = conn.execute(
                        text(
                            f"""
                        SELECT column_name
                        FROM information_schema.columns
                        WHERE table_name = 'product_catalog'
                        AND column_name = '{col}'
                    """
                        )
                    )
                    if result.fetchone():
                        logger.warning(f"⚠ Columna obsoleta '{col}' todavía existe")
                    else:
                        logger.info(f"✓ Columna '{col}' correctamente eliminada")

        return True

    except Exception as e:
        logger.error(f"Error verificando/aplicando migraciones: {str(e)}")
        # En producción, queremos que la app continúe incluso si hay problemas con migraciones
        # pero registramos el error para investigar
        return False


if __name__ == "__main__":
    success = verify_and_apply_migrations()
    sys.exit(0 if success else 1)
