#!/usr/bin/env python3
"""
Script de verificación post-deployment para Render.com

Valida:
1. Variables de entorno necesarias
2. Estado de la base de datos (tablas, migraciones)
3. Health endpoints
4. Datos iniciales (usuarios, farmacia)
5. Genera reporte con estado verde/amarillo/rojo

Uso:
    python backend/scripts/verify_deployment.py
"""

import os
import sys
from datetime import datetime, timezone
from pathlib import Path

import requests

# Añadir el directorio raíz al path
sys.path.insert(0, str(Path(__file__).parent.parent.parent))

from sqlalchemy import create_engine, inspect, text
from sqlalchemy.orm import sessionmaker

# Colores para terminal
GREEN = "\033[92m"
YELLOW = "\033[93m"
RED = "\033[91m"
BLUE = "\033[94m"
RESET = "\033[0m"
BOLD = "\033[1m"


class DeploymentVerifier:
    """Verifica el estado del deployment en Render."""

    def __init__(self):
        self.errors = []
        self.warnings = []
        self.checks_passed = []
        self.engine = None
        self.base_url = None

    def print_header(self, text: str):
        """Imprime encabezado destacado."""
        print(f"\n{BOLD}{BLUE}{'=' * 70}{RESET}")
        print(f"{BOLD}{BLUE}{text.center(70)}{RESET}")
        print(f"{BOLD}{BLUE}{'=' * 70}{RESET}\n")

    def print_section(self, text: str):
        """Imprime sección."""
        print(f"\n{BOLD}🔍 {text}{RESET}")
        print("-" * 70)

    def check_ok(self, message: str):
        """Marca check como exitoso."""
        print(f"{GREEN}✓{RESET} {message}")
        self.checks_passed.append(message)

    def check_warning(self, message: str):
        """Marca advertencia."""
        print(f"{YELLOW}⚠{RESET} {message}")
        self.warnings.append(message)

    def check_error(self, message: str):
        """Marca error."""
        print(f"{RED}✗{RESET} {message}")
        self.errors.append(message)

    def verify_environment_variables(self):
        """Verifica variables de entorno necesarias."""
        self.print_section("Variables de Entorno")

        # Variables obligatorias
        required_vars = {
            "DATABASE_URL": "Conexión a PostgreSQL",
            "SECRET_KEY": "Clave secreta para JWT",
            "ENVIRONMENT": "Entorno de ejecución",
        }

        for var, description in required_vars.items():
            value = os.getenv(var)
            if value:
                # Verificar SECRET_KEY no usa valor de desarrollo
                if var == "SECRET_KEY" and "dev" in value.lower():
                    self.check_warning(f"{var}: Parece usar valor de desarrollo")
                else:
                    self.check_ok(f"{var}: {description} ✓")
            else:
                self.check_error(f"{var}: NO CONFIGURADA (requerida)")

        # Variables opcionales importantes
        optional_vars = {
            "RENDER": "Flag de Render",
            "FRONTEND_URL": "URL del frontend",
            "BASE_URL": "URL base de la aplicación",
            "GOOGLE_CLIENT_ID": "OAuth Google (opcional)",
            "MICROSOFT_CLIENT_ID": "OAuth Microsoft (opcional)",
        }

        print("\nVariables opcionales:")
        for var, description in optional_vars.items():
            value = os.getenv(var)
            if value:
                self.check_ok(f"{var}: {description} ✓")
            else:
                print(f"  {YELLOW}○{RESET} {var}: {description} (no configurada)")

    def verify_database_connection(self):
        """Verifica conexión a base de datos."""
        self.print_section("Conexión a Base de Datos")

        database_url = os.getenv("DATABASE_URL")
        if not database_url:
            self.check_error("DATABASE_URL no configurada")
            return False

        # Render usa postgres:// pero SQLAlchemy requiere postgresql://
        if database_url.startswith("postgres://"):
            database_url = database_url.replace("postgres://", "postgresql://", 1)

        try:
            self.engine = create_engine(database_url)
            with self.engine.connect() as conn:
                result = conn.execute(text("SELECT version()"))
                version = result.fetchone()[0]
                self.check_ok(f"Conexión exitosa: {version.split(',')[0]}")
                return True
        except Exception as e:
            self.check_error(f"Error conectando a base de datos: {e}")
            return False

    def verify_database_tables(self):
        """Verifica que las tablas existan."""
        self.print_section("Tablas de Base de Datos")

        if not self.engine:
            self.check_error("No hay conexión a base de datos")
            return False

        inspector = inspect(self.engine)
        tables = inspector.get_table_names()

        if not tables:
            self.check_error("❌ BASE DE DATOS VACÍA - NO HAY TABLAS")
            print(f"\n{RED}{BOLD}ACCIÓN REQUERIDA:{RESET}")
            print(f"  Ejecutar: {YELLOW}python backend/scripts/reset_database_render.py{RESET}")
            return False

        expected_tables = [
            "users",
            "pharmacies",
            "file_uploads",
            "sales_data",
            "sales_enrichment",
            "product_catalog",
            "nomenclator_local",
            "invitations",
            "audit_logs",
            "system_status",
            "alembic_version",
        ]

        self.check_ok(f"Total de tablas encontradas: {len(tables)}")

        missing_tables = []
        for table in expected_tables:
            if table in tables:
                print(f"  {GREEN}✓{RESET} {table}")
            else:
                missing_tables.append(table)
                print(f"  {RED}✗{RESET} {table} (faltante)")

        if missing_tables:
            self.check_error(f"Faltan {len(missing_tables)} tablas críticas")
            return False
        else:
            self.check_ok("Todas las tablas críticas presentes")
            return True

    def verify_migration_version(self):
        """Verifica versión de migración actual."""
        self.print_section("Migraciones de Alembic")

        if not self.engine:
            return False

        try:
            with self.engine.connect() as conn:
                result = conn.execute(text("SELECT version_num FROM alembic_version"))
                version = result.fetchone()

                if version:
                    self.check_ok(f"Versión de migración: {version[0]}")

                    # Verificar que sea la migración 000
                    if "000" in version[0]:
                        self.check_ok("Usando migración inicial (000_initial_schema)")
                    else:
                        self.check_warning(f"Versión inesperada: {version[0]}")
                    return True
                else:
                    self.check_error("No se encontró versión de migración")
                    return False
        except Exception as e:
            self.check_error(f"Error verificando migraciones: {e}")
            return False

    def verify_health_endpoint(self):
        """Verifica el endpoint /health."""
        self.print_section("Health Endpoint")

        # Determinar URL base
        self.base_url = os.getenv("RENDER_EXTERNAL_URL") or os.getenv("BASE_URL") or "http://localhost:8000"

        try:
            response = requests.get(f"{self.base_url}/health", timeout=10)

            if response.status_code == 200:
                data = response.json()
                self.check_ok(f"Health endpoint respondiendo: {response.status_code}")

                # Verificar campos en la respuesta
                if data.get("status") == "healthy":
                    self.check_ok("Estado del servicio: healthy")
                else:
                    self.check_warning(f"Estado del servicio: {data.get('status')}")

                if "database" in data:
                    self.check_ok(f"Estado de base de datos: {data['database']}")

                if "catalog_products" in data:
                    count = data["catalog_products"]
                    if count > 0:
                        self.check_ok(f"Productos en catálogo: {count:,}")
                    else:
                        self.check_warning("Catálogo vacío (ejecutar sync CIMA si es necesario)")

                return True
            else:
                self.check_error(f"Health endpoint retornó: {response.status_code}")
                if response.status_code == 503:
                    print(f"\n{YELLOW}Posible causa:{RESET} Base de datos sin tablas o servicio iniciando")
                return False

        except requests.RequestException as e:
            self.check_error(f"Error accediendo a health endpoint: {e}")
            print(f"\n{YELLOW}Verificar:{RESET} ¿El servicio está corriendo en {self.base_url}?")
            return False

    def verify_initial_data(self):
        """Verifica que existan datos iniciales."""
        self.print_section("Datos Iniciales")

        if not self.engine:
            return False

        Session = sessionmaker(bind=self.engine)
        session = Session()

        try:
            # Verificar usuarios
            user_count = session.execute(text("SELECT COUNT(*) FROM users")).scalar()
            if user_count > 0:
                self.check_ok(f"Usuarios encontrados: {user_count}")
            else:
                self.check_warning("No hay usuarios - ejecutar init_users.py")

            # Verificar farmacias
            pharmacy_count = session.execute(text("SELECT COUNT(*) FROM pharmacies")).scalar()
            if pharmacy_count > 0:
                self.check_ok(f"Farmacias encontradas: {pharmacy_count}")
            else:
                self.check_warning("No hay farmacias registradas")

            # Verificar catálogo
            catalog_count = session.execute(text("SELECT COUNT(*) FROM product_catalog")).scalar()
            if catalog_count > 0:
                self.check_ok(f"Productos en catálogo: {catalog_count:,}")
            else:
                self.check_warning("Catálogo vacío (sync CIMA pendiente)")

            return True

        except Exception as e:
            self.check_error(f"Error verificando datos iniciales: {e}")
            return False
        finally:
            session.close()

    def print_summary(self):
        """Imprime resumen final."""
        self.print_header("RESUMEN DE VERIFICACIÓN")

        total_checks = len(self.checks_passed) + len(self.warnings) + len(self.errors)

        print(f"\n{GREEN}✓ Checks exitosos: {len(self.checks_passed)}{RESET}")
        print(f"{YELLOW}⚠ Advertencias: {len(self.warnings)}{RESET}")
        print(f"{RED}✗ Errores: {len(self.errors)}{RESET}")
        print(f"\nTotal de verificaciones: {total_checks}")

        if self.errors:
            print(f"\n{RED}{BOLD}ERRORES CRÍTICOS:{RESET}")
            for error in self.errors:
                print(f"  {RED}•{RESET} {error}")

            print(f"\n{BOLD}ACCIONES REQUERIDAS:{RESET}")
            if "BASE DE DATOS VACÍA" in str(self.errors):
                print(f"  1. {YELLOW}python backend/scripts/reset_database_render.py{RESET}")
                print(f"  2. {YELLOW}python backend/scripts/init_users.py{RESET}")

            print(f"\n{RED}{BOLD}🚫 DEPLOYMENT NO ESTÁ LISTO PARA PRODUCCIÓN{RESET}")
            return False

        elif self.warnings:
            print(f"\n{YELLOW}{BOLD}ADVERTENCIAS:{RESET}")
            for warning in self.warnings:
                print(f"  {YELLOW}•{RESET} {warning}")

            print(f"\n{YELLOW}{BOLD}⚠️  DEPLOYMENT FUNCIONAL CON ADVERTENCIAS{RESET}")
            return True

        else:
            print(f"\n{GREEN}{BOLD}✅ DEPLOYMENT EXITOSO - TODO OK{RESET}")
            return True

    def run(self):
        """Ejecuta todas las verificaciones."""
        self.print_header("🔧 VERIFICACIÓN DE DEPLOYMENT - Render.com")

        print(f"Fecha: {datetime.now(timezone.utc).strftime('%Y-%m-%d %H:%M:%S UTC')}")
        print(f"Entorno: {os.getenv('ENVIRONMENT', 'unknown')}")
        print(f"Render: {os.getenv('RENDER', 'false')}")

        # Ejecutar verificaciones
        self.verify_environment_variables()

        if self.verify_database_connection():
            self.verify_database_tables()
            self.verify_migration_version()
            self.verify_initial_data()

        self.verify_health_endpoint()

        # Resumen final
        success = self.print_summary()

        # Exit code
        sys.exit(0 if success else 1)


if __name__ == "__main__":
    verifier = DeploymentVerifier()
    verifier.run()
