#!/usr/bin/env python3
"""
Script de verificación de clasificación de productos.

Verifica:
1. División jerárquica: Prescripción vs Venta Libre
2. Categorías de prescripción bien asignadas
3. Consistencia entre product_catalog y sales_enrichment

Uso:
    python scripts/verify_classification.py
    python scripts/verify_classification.py --detailed
"""
import argparse
import sys
from pathlib import Path
from datetime import datetime

# Añadir raíz del proyecto al PYTHONPATH
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.database import SessionLocal
from app.models.product_catalog import ProductCatalog
from app.models.sales_enrichment import SalesEnrichment
from app.models.enums import PrescriptionCategory
from sqlalchemy import func, and_, or_


def print_header(title):
    """Imprime un header visual"""
    print("\n" + "=" * 80)
    print(f"  {title}")
    print("=" * 80)


def verify_hierarchical_division(db):
    """Verifica la división jerárquica Prescripción vs Venta Libre"""
    print_header("NIVEL 1: DIVISIÓN JERÁRQUICA (Prescripción vs Venta Libre)")

    # Total de productos
    total_products = db.query(ProductCatalog).count()

    # Productos con categoría (Prescripción)
    prescription_count = db.query(ProductCatalog).filter(
        ProductCatalog.xfarma_prescription_category.isnot(None)
    ).count()

    # Productos sin categoría (Venta Libre)
    free_sale_count = db.query(ProductCatalog).filter(
        ProductCatalog.xfarma_prescription_category.is_(None)
    ).count()

    print(f"\nTOTAL PRODUCTOS EN CATÁLOGO: {total_products:,}")
    print("-" * 80)
    print(f"  ✅ PRESCRIPCIÓN (con categoría):        {prescription_count:>8,}  ({prescription_count/total_products*100:>5.1f}%)")
    print(f"  ✅ VENTA LIBRE (sin categoría):         {free_sale_count:>8,}  ({free_sale_count/total_products*100:>5.1f}%)")

    # Verificar suma
    assert prescription_count + free_sale_count == total_products, "ERROR: Suma no coincide"

    return {
        'total': total_products,
        'prescription': prescription_count,
        'free_sale': free_sale_count
    }


def verify_prescription_categories(db):
    """Verifica la distribución de categorías de prescripción"""
    print_header("NIVEL 2: CATEGORÍAS DE PRESCRIPCIÓN (14 categorías)")

    # Contar por categoría
    category_counts = db.query(
        ProductCatalog.xfarma_prescription_category,
        func.count(ProductCatalog.id)
    ).filter(
        ProductCatalog.xfarma_prescription_category.isnot(None)
    ).group_by(
        ProductCatalog.xfarma_prescription_category
    ).all()

    # Ordenar por cantidad descendente
    category_counts = sorted(category_counts, key=lambda x: x[1], reverse=True)

    print("\nDISTRIBUCIÓN DE CATEGORÍAS:")
    print("-" * 80)

    total_categorized = 0
    for category, count in category_counts:
        total_categorized += count
        print(f"  {category:.<60} {count:>8,}")

    print("-" * 80)
    print(f"  {'TOTAL PRODUCTOS DE PRESCRIPCIÓN':.<60} {total_categorized:>8,}")

    # Verificar categorías vacías
    all_categories = {cat.value for cat in PrescriptionCategory}
    used_categories = {cat for cat, _ in category_counts}
    empty_categories = all_categories - used_categories

    if empty_categories:
        print("\n⚠️  CATEGORÍAS SIN PRODUCTOS:")
        for cat in sorted(empty_categories):
            print(f"     - {cat}")

    return category_counts


def verify_sales_enrichment_consistency(db):
    """Verifica consistencia entre product_catalog y sales_enrichment"""
    print_header("NIVEL 3: CONSISTENCIA SALES_ENRICHMENT")

    # Contar por product_type en sales_enrichment
    enrichment_types = db.query(
        SalesEnrichment.product_type,
        func.count(SalesEnrichment.id)
    ).group_by(
        SalesEnrichment.product_type
    ).all()

    print("\nDISTRIBUCIÓN EN SALES_ENRICHMENT (product_type):")
    print("-" * 80)
    for ptype, count in sorted(enrichment_types, key=lambda x: x[1], reverse=True):
        display_type = ptype if ptype else "NULL"
        print(f"  {display_type:.<60} {count:>8,}")

    # Verificar inconsistencias
    print("\n🔍 VERIFICANDO INCONSISTENCIAS...")

    # Query para encontrar enriquecimientos con tipo incorrecto
    inconsistencies = db.query(
        SalesEnrichment.id,
        SalesEnrichment.product_type,
        ProductCatalog.xfarma_prescription_category
    ).join(
        ProductCatalog,
        SalesEnrichment.product_catalog_id == ProductCatalog.id
    ).filter(
        or_(
            # Caso 1: product_type='prescription' pero sin categoría
            and_(
                SalesEnrichment.product_type == 'prescription',
                ProductCatalog.xfarma_prescription_category.is_(None)
            ),
            # Caso 2: product_type='free_sale' pero tiene categoría
            and_(
                SalesEnrichment.product_type == 'free_sale',
                ProductCatalog.xfarma_prescription_category.isnot(None)
            ),
            # Caso 3: product_type='otc' (legacy - debe actualizarse)
            SalesEnrichment.product_type == 'otc'
        )
    ).limit(100).all()

    if inconsistencies:
        print(f"\n❌ ENCONTRADAS {len(inconsistencies)} INCONSISTENCIAS (mostrando primeras 100):")
        print("-" * 80)
        for enrich_id, ptype, category in inconsistencies[:10]:
            print(f"  ID: {enrich_id} | Type: {ptype} | Category: {category}")

        if len(inconsistencies) > 10:
            print(f"  ... y {len(inconsistencies) - 10} más")

        return False
    else:
        print("  ✅ No se encontraron inconsistencias")
        return True


def identify_unclassified_prescription_products(db, detailed=False):
    """Identifica productos que deberían estar clasificados pero no lo están"""
    print_header("NIVEL 4: PRODUCTOS POTENCIALMENTE MAL CLASIFICADOS")

    # Productos en nomenclator/CIMA pero sin categoría
    unclassified_nomenclator = db.query(ProductCatalog).filter(
        and_(
            or_(
                ProductCatalog.nomen_codigo_homogeneo.isnot(None),
                ProductCatalog.cima_requiere_receta == True
            ),
            ProductCatalog.xfarma_prescription_category.is_(None)
        )
    ).limit(100).all()

    if unclassified_nomenclator:
        print(f"\n⚠️  PRODUCTOS EN NOMENCLATOR/CIMA SIN CATEGORÍA: {len(unclassified_nomenclator)}")
        print("-" * 80)

        if detailed:
            for product in unclassified_nomenclator[:20]:
                print(f"  CN: {product.national_code:>8} | {product.nomen_nombre or product.cima_nombre_comercial}")
                if product.nomen_codigo_homogeneo:
                    print(f"     → Código homogéneo: {product.nomen_codigo_homogeneo}")
                if product.cima_requiere_receta:
                    print(f"     → Requiere receta: Sí")
        else:
            print("  (Usar --detailed para ver lista completa)")
    else:
        print("\n  ✅ Todos los productos de nomenclator/CIMA están clasificados")

    return len(unclassified_nomenclator)


def suggest_reclassification(db):
    """Sugiere acciones de reclasificación"""
    print_header("RECOMENDACIONES")

    # Contar productos legacy con 'otc'
    legacy_otc = db.query(SalesEnrichment).filter(
        SalesEnrichment.product_type == 'otc'
    ).count()

    if legacy_otc > 0:
        print(f"\n🔄 ACTUALIZACIÓN REQUERIDA:")
        print(f"  - {legacy_otc:,} registros en sales_enrichment con product_type='otc' (legacy)")
        print(f"  - Acción: Re-ejecutar enriquecimiento para actualizar a 'free_sale'")
        print(f"  - Comando: docker-compose exec backend python scripts/reenrich_all.py")

    # Productos sin clasificar
    unclassified = db.query(ProductCatalog).filter(
        and_(
            or_(
                ProductCatalog.nomen_codigo_homogeneo.isnot(None),
                ProductCatalog.cima_requiere_receta == True
            ),
            ProductCatalog.xfarma_prescription_category.is_(None)
        )
    ).count()

    if unclassified > 0:
        print(f"\n📋 CLASIFICACIÓN PENDIENTE:")
        print(f"  - {unclassified:,} productos de nomenclator/CIMA sin categoría")
        print(f"  - Acción: Re-ejecutar clasificación")
        print(f"  - Comando: docker-compose exec backend python scripts/classify_catalog.py")


def main():
    parser = argparse.ArgumentParser(
        description="Verificar clasificación de productos (Prescripción vs Venta Libre)"
    )
    parser.add_argument(
        '--detailed',
        action='store_true',
        help="Mostrar detalles completos de productos mal clasificados"
    )
    args = parser.parse_args()

    print("=" * 80)
    print("VERIFICACIÓN DE CLASIFICACIÓN DE PRODUCTOS")
    print("=" * 80)
    print(f"Timestamp: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    print(f"Modo: {'DETALLADO' if args.detailed else 'RESUMEN'}")
    print("=" * 80)

    db = SessionLocal()

    try:
        # Nivel 1: División jerárquica
        hierarchical_stats = verify_hierarchical_division(db)

        # Nivel 2: Categorías de prescripción
        category_distribution = verify_prescription_categories(db)

        # Nivel 3: Consistencia sales_enrichment
        is_consistent = verify_sales_enrichment_consistency(db)

        # Nivel 4: Productos mal clasificados
        unclassified_count = identify_unclassified_prescription_products(db, args.detailed)

        # Recomendaciones
        suggest_reclassification(db)

        # Resumen final
        print_header("RESUMEN")
        print(f"\n  Total productos: {hierarchical_stats['total']:,}")
        print(f"  Prescripción: {hierarchical_stats['prescription']:,}")
        print(f"  Venta Libre: {hierarchical_stats['free_sale']:,}")
        print(f"  Categorías con productos: {len(category_distribution)}/14")
        print(f"  Consistencia sales_enrichment: {'✅ OK' if is_consistent else '❌ REVISAR'}")
        print(f"  Productos sin clasificar: {unclassified_count:,}")
        print("=" * 80)

    except Exception as e:
        print(f"\n❌ ERROR: {e}", file=sys.stderr)
        import traceback
        traceback.print_exc()
        sys.exit(1)
    finally:
        db.close()


if __name__ == "__main__":
    main()
