#!/usr/bin/env python
"""
Validación de la migración 000_initial_schema.py

Este script verifica que:
1. Todos los modelos se pueden importar correctamente
2. La migración puede ejecutarse sin errores
3. Todas las tablas esperadas se crearán
4. Los modelos son thread-safe (extend_existing=True)

Ejecutar ANTES de deployment en Render.
"""

import sys
from pathlib import Path

# Añadir el directorio backend al path
backend_dir = Path(__file__).parent.parent
sys.path.insert(0, str(backend_dir))


def validate_model_imports():
    """Validar que todos los modelos se pueden importar correctamente"""
    print("🔍 Validando imports de modelos...")

    try:
        from app.database import Base

        # System health models
        # Auth & Security models
        # Analytics models
        # Sales pipeline models
        # Core models
        from app.models import (
            AuditLog,
            DeveloperLog,
            FileUpload,
            HomogeneousGroup,
            HomogeneousGroupMaster,
            Invitation,
            NomenclatorLocal,
            PerformanceSnapshot,
            Pharmacy,
            PharmacyHomogeneousMetrics,
            PharmacyPartner,
            ProductCatalog,
            SalesData,
            SalesEnrichment,
            SystemAlert,
            SystemComponent,
            SystemHealthMetric,
            SystemStatus,
            SystemStatusEnum,
            User,
        )

        print("✅ Todos los modelos importados correctamente")
        return True

    except ImportError as e:
        print(f"❌ Error importando modelos: {e}")
        return False


def validate_base_metadata():
    """Validar que Base.metadata conoce todas las tablas"""
    print("\n🔍 Validando Base.metadata...")

    try:
        from app.database import Base

        # Importar todos los modelos para que se registren en Base.metadata

        # Tablas esperadas
        expected_tables = {
            "users",
            "pharmacies",
            "sales_data",
            "sales_enrichment",
            "product_catalog",
            "nomenclator_local",
            "pharmacy_partners",
            "file_uploads",
            "invitations",
            "audit_logs",
            "system_status",
            "system_health_metrics",
            "system_alerts",
            "developer_logs",
            "performance_snapshots",
            "homogeneous_groups",
            "homogeneous_group_masters",
            "pharmacy_homogeneous_metrics",
        }

        # Obtener tablas registradas en metadata
        registered_tables = set(Base.metadata.tables.keys())

        print(f"   Tablas registradas: {len(registered_tables)}")
        print(f"   Tablas esperadas: {len(expected_tables)}")

        # Verificar que todas las tablas esperadas están registradas
        missing_tables = expected_tables - registered_tables
        extra_tables = registered_tables - expected_tables

        if missing_tables:
            print(f"❌ Tablas faltantes: {missing_tables}")
            return False

        if extra_tables:
            print(f"⚠️  Tablas extra (no esperadas): {extra_tables}")

        print("✅ Base.metadata contiene todas las tablas esperadas")

        # Mostrar lista completa de tablas
        print("\n📋 Lista de tablas que se crearán:")
        for i, table_name in enumerate(sorted(registered_tables), 1):
            print(f"   {i:2d}. {table_name}")

        return True

    except Exception as e:
        print(f"❌ Error validando metadata: {e}")
        import traceback

        traceback.print_exc()
        return False


def validate_thread_safety():
    """Validar que todos los modelos tienen extend_existing=True"""
    print("\n🔍 Validando thread-safety (extend_existing)...")

    try:

        # Importar todos los modelos
        from app.models import (
            AuditLog,
            DeveloperLog,
            FileUpload,
            HomogeneousGroup,
            HomogeneousGroupMaster,
            Invitation,
            NomenclatorLocal,
            PerformanceSnapshot,
            Pharmacy,
            PharmacyHomogeneousMetrics,
            PharmacyPartner,
            ProductCatalog,
            SalesData,
            SalesEnrichment,
            SystemAlert,
            SystemHealthMetric,
            SystemStatus,
            User,
        )

        models_to_check = [
            Pharmacy,
            User,
            FileUpload,
            SalesData,
            SalesEnrichment,
            ProductCatalog,
            NomenclatorLocal,
            PharmacyPartner,
            HomogeneousGroup,
            HomogeneousGroupMaster,
            PharmacyHomogeneousMetrics,
            Invitation,
            AuditLog,
            SystemStatus,
            SystemHealthMetric,
            SystemAlert,
            DeveloperLog,
            PerformanceSnapshot,
        ]

        issues_found = []

        for model in models_to_check:
            # Verificar __table_args__
            if hasattr(model, "__table_args__"):
                table_args = model.__table_args__

                # Puede ser dict o tuple
                if isinstance(table_args, dict):
                    if not table_args.get("extend_existing", False):
                        issues_found.append(f"{model.__name__}: falta extend_existing=True")
                elif isinstance(table_args, tuple):
                    # El último elemento debe ser un dict con extend_existing
                    if (
                        len(table_args) > 0
                        and isinstance(table_args[-1], dict)
                        and not table_args[-1].get("extend_existing", False)
                    ):
                        issues_found.append(f"{model.__name__}: falta extend_existing=True")
                    elif not any(isinstance(arg, dict) and arg.get("extend_existing", False) for arg in table_args):
                        issues_found.append(f"{model.__name__}: falta extend_existing=True")
            else:
                issues_found.append(f"{model.__name__}: falta __table_args__")

        if issues_found:
            print("❌ Problemas de thread-safety encontrados:")
            for issue in issues_found:
                print(f"   - {issue}")
            return False

        print("✅ Todos los modelos tienen extend_existing=True (thread-safe)")
        return True

    except Exception as e:
        print(f"❌ Error validando thread-safety: {e}")
        import traceback

        traceback.print_exc()
        return False


def validate_migration_file():
    """Validar que el archivo de migración existe y es válido"""
    print("\n🔍 Validando archivo de migración...")

    migration_file = backend_dir / "alembic" / "versions" / "000_initial_schema.py"

    if not migration_file.exists():
        print(f"❌ Archivo de migración no encontrado: {migration_file}")
        return False

    print(f"✅ Archivo de migración encontrado: {migration_file}")

    # Intentar importar la migración
    try:
        # Añadir directorio de versiones al path
        versions_dir = migration_file.parent
        sys.path.insert(0, str(versions_dir))

        # Importar módulo de migración (sin ejecutar)
        import importlib.util

        spec = importlib.util.spec_from_file_location("migration_000", migration_file)
        migration_module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(migration_module)

        # Verificar que tiene funciones upgrade y downgrade
        if not hasattr(migration_module, "upgrade"):
            print("❌ Falta función upgrade() en migración")
            return False

        if not hasattr(migration_module, "downgrade"):
            print("❌ Falta función downgrade() en migración")
            return False

        print("✅ Migración tiene funciones upgrade() y downgrade()")
        return True

    except Exception as e:
        print(f"❌ Error validando migración: {e}")
        import traceback

        traceback.print_exc()
        return False


def main():
    """Ejecutar todas las validaciones"""
    print("=" * 70)
    print("🚀 VALIDACIÓN DE MIGRACIÓN 000_initial_schema.py")
    print("=" * 70)

    results = []

    # 1. Validar imports de modelos
    results.append(("Imports de modelos", validate_model_imports()))

    # 2. Validar Base.metadata
    results.append(("Base.metadata", validate_base_metadata()))

    # 3. Validar thread-safety
    results.append(("Thread-safety", validate_thread_safety()))

    # 4. Validar archivo de migración
    results.append(("Archivo de migración", validate_migration_file()))

    # Resumen
    print("\n" + "=" * 70)
    print("📊 RESUMEN DE VALIDACIÓN")
    print("=" * 70)

    all_passed = True
    for test_name, passed in results:
        status = "✅ PASS" if passed else "❌ FAIL"
        print(f"{status:10} - {test_name}")
        if not passed:
            all_passed = False

    print("=" * 70)

    if all_passed:
        print("✅ TODAS LAS VALIDACIONES PASARON")
        print("🚀 La migración está lista para deployment en Render")
        return 0
    else:
        print("❌ ALGUNAS VALIDACIONES FALLARON")
        print("⚠️  Corregir los errores antes de deployment")
        return 1


if __name__ == "__main__":
    sys.exit(main())
