﻿#!/usr/bin/env python
"""
Script optimizado para actualizar el catálogo CIMA con procesamiento por lotes.
Ventajas sobre el enfoque anterior:
- Uso de memoria constante (procesa por lotes de 1000)
- Progreso visible en tiempo real en la BD
- Resistente a fallos (no pierde trabajo completado)
- Commits incrementales más eficientes
"""

import logging
import os
import sys
import time
from typing import Any, Dict, List

# Añadir el directorio padre al path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

# Configurar para producción
os.environ["ENVIRONMENT"] = "production"

import requests

from app.database import SessionLocal
from app.external_data.cima_integration import CIMAIntegrationService
from app.models import ProductCatalog, SystemStatus
from app.utils.datetime_utils import utc_now

# Configurar logging
logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(name)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


class OptimizedCIMAUpdater:
    """Actualizador optimizado con procesamiento por streaming"""

    def __init__(self, batch_size: int = 1000):
        self.batch_size = batch_size
        self.db = SessionLocal()
        self.service = CIMAIntegrationService()
        self.updated = 0
        self.created = 0
        self.errors = 0
        self.processed = 0

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.db.close()

    def update_system_status(self, status: str, progress: int, message: str, total_items: int = None):
        """Actualiza el estado del sistema en tiempo real"""
        cima_status = self.db.query(SystemStatus).filter_by(component="CIMA").first()
        if not cima_status:
            cima_status = SystemStatus(component="CIMA")
            self.db.add(cima_status)

        cima_status.status = status
        cima_status.progress = progress
        cima_status.message = message

        if total_items:
            cima_status.total_items = total_items
        if self.processed > 0:
            cima_status.processed_items = self.processed

        # Actualizar detalles con estadísticas en tiempo real
        cima_status.details = {
            "actualizados": self.updated,
            "creados": self.created,
            "errores": self.errors,
            "procesados": self.processed,
            "batch_size": self.batch_size,
        }

        if status == "INITIALIZING":
            cima_status.started_at = utc_now()
        elif status == "COMPLETED":
            cima_status.last_success_at = utc_now()
        elif status == "ERROR":
            cima_status.last_error_at = utc_now()

        self.db.commit()

    def fetch_presentations_streaming(self):
        """
        Generador que descarga presentaciones en lotes.
        Más eficiente que descargar todo de una vez.
        """
        page = 1
        page_size = self.batch_size
        total_fetched = 0

        # URL base de la API CIMA
        base_url = "https://cima.aemps.es/cima/rest/presentaciones"

        while True:
            try:
                # Construir URL con paginación
                params = {"pageSize": page_size, "pageNum": page}

                logger.info(f"📥 Descargando página {page} (batch de {page_size} productos)...")

                response = requests.get(base_url, params=params, timeout=30)

                if response.status_code != 200:
                    logger.warning(f"⚠️ Error en página {page}: HTTP {response.status_code}")
                    break

                data = response.json()
                presentations = data.get("resultados", [])

                if not presentations:
                    logger.info(f"✅ Descarga completa. Total: {total_fetched} productos")
                    break

                total_fetched += len(presentations)
                yield presentations

                # Si recibimos menos del tamaño de página, es la última
                if len(presentations) < page_size:
                    logger.info(f"✅ Última página. Total descargado: {total_fetched}")
                    break

                page += 1

                # Pequeña pausa para no saturar la API
                time.sleep(0.1)

            except Exception as e:
                logger.error(f"❌ Error descargando página {page}: {str(e)}")
                break

    def process_batch(self, presentations: List[Dict[str, Any]]):
        """Procesa y guarda un lote de presentaciones"""
        batch_updated = 0
        batch_created = 0
        batch_errors = 0

        for pres in presentations:
            try:
                cn = pres.get("cn")
                if not cn:
                    continue

                # Buscar producto en catálogo
                product = self.db.query(ProductCatalog).filter_by(national_code=str(cn)).first()

                if product:
                    # Actualizar con datos CIMA
                    product.cima_nombre_comercial = pres.get("nombre")
                    product.cima_laboratorio_titular = pres.get("labtitular")
                    product.cima_principios_activos = pres.get("pactivos")
                    product.cima_requiere_receta = pres.get("receta")
                    product.cima_estado_registro = "Comercializado" if pres.get("comerc") else "No comercializado"

                    # Marcar como genérico si contiene EFG
                    nombre = pres.get("nombre", "")
                    product.cima_es_generico = "EFG" in nombre.upper()

                    # Actualizar fuente de datos
                    if product.data_sources and "cima" not in product.data_sources.lower():
                        product.data_sources = f"{product.data_sources},cima"
                    elif not product.data_sources:
                        product.data_sources = "cima"

                    product.sync_status = "SINCRONIZADO"
                    product.updated_at = utc_now()
                    batch_updated += 1
                else:
                    # Crear nuevo producto si no existe
                    product = ProductCatalog(
                        national_code=str(cn),
                        cima_nombre_comercial=pres.get("nombre"),
                        cima_laboratorio_titular=pres.get("labtitular"),
                        cima_principios_activos=pres.get("pactivos"),
                        cima_requiere_receta=pres.get("receta"),
                        cima_estado_registro=("Comercializado" if pres.get("comerc") else "No comercializado"),
                        cima_es_generico="EFG" in (pres.get("nombre", "")).upper(),
                        data_sources="cima",
                        sync_status="SINCRONIZADO",
                        created_at=utc_now(),
                        updated_at=utc_now(),
                    )
                    self.db.add(product)
                    batch_created += 1

            except Exception as e:
                batch_errors += 1
                logger.error(f"Error procesando CN {cn}: {str(e)}")
                continue

        # Commit del lote
        try:
            self.db.commit()
            self.updated += batch_updated
            self.created += batch_created
            self.errors += batch_errors
            self.processed += len(presentations)

            logger.info(
                f"📊 Lote procesado | "
                f"Actualizados: {batch_updated} | "
                f"Creados: {batch_created} | "
                f"Total acumulado: {self.processed}"
            )

        except Exception as e:
            logger.error(f"❌ Error en commit del lote: {str(e)}")
            self.db.rollback()
            self.errors += len(presentations)

    def run(self):
        """Ejecuta la actualización completa con streaming"""
        try:
            logger.info("=" * 60)
            logger.info("INICIANDO ACTUALIZACIÓN OPTIMIZADA DEL CATÁLOGO CIMA")
            logger.info(f"Procesamiento por lotes de {self.batch_size} productos")
            logger.info("=" * 60)

            # Actualizar estado inicial
            self.update_system_status("INITIALIZING", 0, "Iniciando descarga y procesamiento por lotes...")

            # Estimar total (basado en conocimiento previo)
            estimated_total = 67000

            # Procesar por lotes con streaming
            batch_num = 0
            for batch in self.fetch_presentations_streaming():
                batch_num += 1

                # Actualizar estado antes de procesar
                progress = min(int((self.processed / estimated_total) * 100), 99)
                self.update_system_status(
                    "PROCESSING",
                    progress,
                    f"Procesando lote {batch_num} ({self.processed}/{estimated_total} aprox.)",
                    estimated_total,
                )

                # Procesar y guardar el lote
                self.process_batch(batch)

                # Actualizar progreso después de procesar
                progress = min(int((self.processed / estimated_total) * 100), 99)
                self.update_system_status(
                    "PROCESSING",
                    progress,
                    f"Lote {batch_num} completado. Procesados: {self.processed}",
                    self.processed,  # Actualizar con total real
                )

            # Estado final
            self.update_system_status(
                "COMPLETED", 100, f"Completado: {self.updated} actualizados, {self.created} creados", self.processed
            )

            logger.info("=" * 60)
            logger.info("ACTUALIZACIÓN COMPLETADA")
            logger.info(f"✅ Productos actualizados: {self.updated}")
            logger.info(f"✅ Productos creados: {self.created}")
            logger.info(f"📊 Total procesados: {self.processed}")
            if self.errors > 0:
                logger.info(f"⚠️  Errores encontrados: {self.errors}")
            logger.info("=" * 60)

            return {
                "success": True,
                "updated": self.updated,
                "created": self.created,
                "errors": self.errors,
                "total": self.processed,
            }

        except Exception as e:
            logger.error(f"❌ Error crítico: {str(e)}")

            # Actualizar estado de error
            self.update_system_status(
                "ERROR", int((self.processed / 67000) * 100) if self.processed > 0 else 0, f"Error: {str(e)}"
            )

            import traceback

            traceback.print_exc()

            return {
                "success": False,
                "error": str(e),
                "partial_results": {"updated": self.updated, "created": self.created, "processed": self.processed},
            }


if __name__ == "__main__":
    # Permitir configurar batch_size desde línea de comandos
    batch_size = 1000
    if len(sys.argv) > 1:
        try:
            batch_size = int(sys.argv[1])
            logger.info(f"Usando batch_size personalizado: {batch_size}")
        except:
            pass

    with OptimizedCIMAUpdater(batch_size=batch_size) as updater:
        result = updater.run()

        if result["success"]:
            sys.exit(0)
        else:
            sys.exit(1)
