﻿#!/usr/bin/env python
"""
Script para actualizar el catálogo CIMA de forma completa
"""

import logging
import os
import sys

# Añadir el directorio padre al path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

# Configurar para producción
os.environ["ENVIRONMENT"] = "production"


from app.database import SessionLocal
from app.external_data.cima_integration import CIMAIntegrationService
from app.models import ProductCatalog, SystemStatus
from app.utils.datetime_utils import utc_now

# Configurar logging
logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(name)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


def update_cima_catalog():
    """Actualiza el catálogo con datos de CIMA usando streaming"""

    db = SessionLocal()
    service = CIMAIntegrationService()

    try:
        logger.info("=" * 60)
        logger.info("INICIANDO ACTUALIZACIÓN OPTIMIZADA DEL CATÁLOGO CIMA")
        logger.info("Usando procesamiento por streaming (más eficiente)")
        logger.info("=" * 60)

        # Función para actualizar el estado del sistema
        def update_system_status(status: str, progress: int, message: str, stats: dict):
            try:
                cima_status = db.query(SystemStatus).filter_by(component="CIMA").first()
                if not cima_status:
                    cima_status = SystemStatus(component="CIMA")
                    db.add(cima_status)

                cima_status.status = status
                cima_status.progress = progress
                cima_status.message = message

                # Actualizar detalles con estadísticas
                if stats:
                    cima_status.total_items = stats.get("total_processed", 0)
                    cima_status.processed_items = stats.get("total_processed", 0)
                    cima_status.error_count = stats.get("errors", 0)

                if status == "INITIALIZING":
                    cima_status.started_at = utc_now()
                elif status == "COMPLETED":
                    cima_status.last_success_at = utc_now()
                elif status == "ERROR":
                    cima_status.last_error_at = utc_now()

                db.commit()
            except Exception as e:
                logger.error(f"Error actualizando estado del sistema: {str(e)}")

        # Usar el nuevo método de sincronización con streaming
        if hasattr(service, "sync_catalog_streaming"):
            logger.info("📥 Iniciando sincronización con streaming...")
            logger.info("⏳ Los datos se procesarán por lotes de 200 productos...")

            # Ejecutar sincronización con callback para actualización de estado
            result = service.sync_catalog_streaming(db, update_system_status)

            if result.get("error"):
                logger.error(f"❌ Error durante sincronización: {result['error']}")
            else:
                logger.info("=" * 60)
                logger.info("ACTUALIZACIÓN COMPLETADA")
                logger.info(f"✅ Total procesados: {result.get('total_processed', 0)}")
                logger.info(f"✅ Productos creados: {result.get('created', 'N/A')}")
                logger.info(f"✅ Productos actualizados: {result.get('updated', 'N/A')}")
                if result.get("errors", 0) > 0:
                    logger.info(f"⚠️  Errores encontrados: {result['errors']}")
                logger.info("=" * 60)
        else:
            # Fallback al método antiguo si no existe el nuevo
            logger.warning("⚠️  Método streaming no disponible, usando método legacy...")

            # [Código antiguo como fallback - mantener por compatibilidad]
            presentations = service.fetch_all_presentations_batch()

            if not presentations:
                logger.error("❌ No se pudieron descargar presentaciones")
                update_system_status("ERROR", 0, "No se pudieron descargar datos de CIMA", {})
                return

            logger.info(f"✅ Descargadas {len(presentations)} presentaciones de CIMA")

            # Procesar con el método antiguo
            updated = 0
            created = 0
            errors = 0
            batch_size = 100

            for i, pres in enumerate(presentations, 1):
                try:
                    cn = pres.get("cn")
                    if not cn:
                        continue

                    # Buscar producto en catálogo
                    product = db.query(ProductCatalog).filter_by(national_code=str(cn)).first()

                    if product:
                        # Actualizar con datos CIMA
                        product.cima_nombre_comercial = pres.get("nombre")
                        product.cima_laboratorio_titular = pres.get("labtitular")
                        product.cima_principios_activos = pres.get("pactivos")
                        product.cima_requiere_receta = pres.get("receta")
                        product.cima_estado_registro = "Comercializado" if pres.get("comerc") else "No comercializado"

                        # Marcar el medicamento como genérico si contiene EFG
                        nombre = pres.get("nombre", "")
                        product.cima_es_generico = "EFG" in nombre.upper()

                        # Actualizar fuente de datos
                        if product.data_sources and "cima" not in product.data_sources.lower():
                            product.data_sources = f"{product.data_sources},cima"
                        elif not product.data_sources:
                            product.data_sources = "cima"

                        product.sync_status = "SINCRONIZADO"
                        product.updated_at = utc_now()
                        updated += 1
                    else:
                        # Crear nuevo producto si no existe
                        product = ProductCatalog(
                            national_code=str(cn),
                            cima_nombre_comercial=pres.get("nombre"),
                            cima_laboratorio_titular=pres.get("labtitular"),
                            cima_principios_activos=pres.get("pactivos"),
                            cima_requiere_receta=pres.get("receta"),
                            cima_estado_registro="Comercializado" if pres.get("comerc") else "No comercializado",
                            cima_es_generico="EFG" in (pres.get("nombre", "")).upper(),
                            data_sources="cima",
                            sync_status="SINCRONIZADO",
                            created_at=utc_now(),
                            updated_at=utc_now(),
                        )
                        db.add(product)
                        created += 1

                    # Commit por lotes
                    if i % batch_size == 0:
                        db.commit()

                        # Actualizar progreso
                        progress = int((i / len(presentations)) * 100)
                        stats = {"total_processed": i, "created": created, "updated": updated, "errors": errors}
                        update_system_status("PROCESSING", progress, f"Procesando... ({i}/{len(presentations)})", stats)

                        logger.info(
                            f"📊 Progreso: {progress}% | Procesados: {i}/{len(presentations)} | Actualizados: {updated} | Creados: {created}"
                        )

                except Exception as e:
                    errors += 1
                    logger.error(f"Error procesando CN {cn}: {str(e)}")
                    continue

            # Commit final
            db.commit()

            # Actualizar estado final
            final_stats = {
                "total_processed": len(presentations),
                "created": created,
                "updated": updated,
                "errors": errors,
            }
            update_system_status(
                "COMPLETED", 100, f"Completado: {updated} actualizados, {created} creados", final_stats
            )

            logger.info("=" * 60)
            logger.info("ACTUALIZACIÓN COMPLETADA")
            logger.info(f"✅ Productos actualizados: {updated}")
            logger.info(f"✅ Productos creados: {created}")
            if errors > 0:
                logger.info(f"⚠️  Errores encontrados: {errors}")
            logger.info("=" * 60)

    except Exception as e:
        logger.error(f"❌ Error crítico: {str(e)}")

        # Actualizar estado de error
        if "cima_status" in locals():
            cima_status.status = "ERROR"
            cima_status.message = f"Error: {str(e)}"
            cima_status.last_error_at = utc_now()
            db.commit()

        import traceback

        traceback.print_exc()

    finally:
        db.close()


if __name__ == "__main__":
    update_cima_catalog()
