#!/usr/bin/env python3
"""
Script de prueba para verificar Sync Incremental de CIMAVet.
Sprint 3 - Issue #354: Integración CIMAVet con estrategia incremental.

Prueba con 9 productos veterinarios reales encontrados en ventas de dgruiz@gmail.com:
- SYNULOX (3 presentaciones)
- APOQUEL
- FRONTLINE
- RIMADYL
- BRAVECTO (2 presentaciones)
- NEXGARD
"""

import asyncio
import sys
from pathlib import Path

# Add backend to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.services.cimavet_sync_service import cimavet_sync_service
from app.database import SessionLocal
import structlog

logger = structlog.get_logger(__name__)

# Códigos nacionales de productos veterinarios reales en ventas
REAL_VET_CODES = [
    "572240",  # SYNULOX 500/100 mg
    "572239",  # SYNULOX 50/12.5 mg
    "572301",  # SYNULOX 200/50 mg
    "581781",  # APOQUEL 5.4 mg
    "578403",  # FRONTLINE Combo
    "572544",  # RIMADYL 100 mg
    "589610",  # BRAVECTO 500 mg
    "589612",  # BRAVECTO 1000 mg
    "580891",  # NEXGARD 28.3 mg
]


async def test_sync_incremental():
    """Prueba de sincronización incremental con productos reales."""
    print("=" * 80)
    print("🧪 TEST: Sync Incremental CIMAVet (Sprint 3 - Issue #354)")
    print("=" * 80)

    print(f"\n📋 Productos a sincronizar: {len(REAL_VET_CODES)}")
    for i, code in enumerate(REAL_VET_CODES, 1):
        print(f"  {i}. CN: {code}")

    db = SessionLocal()
    try:
        print(f"\n🔄 Ejecutando sync_cimavet_catalog()...")
        print(f"   Estrategia: Sync Incremental (individual CN queries)")
        print(f"   API: {cimavet_sync_service.api_base_url}")

        result = await cimavet_sync_service.sync_cimavet_catalog(
            db, force=True, national_codes=REAL_VET_CODES
        )

        print(f"\n📊 Resultado de sincronización:")
        print(f"  - Status: {result.get('status')}")

        if result.get("stats"):
            stats = result["stats"]
            print(f"\n  📈 Estadísticas:")
            print(f"     - Procesados: {stats.get('processed', 0)}")
            print(f"     - Creados: {stats.get('created', 0)}")
            print(f"     - Actualizados: {stats.get('updated', 0)}")
            print(f"     - Errores: {stats.get('errors', 0)}")
            print(f"     - No encontrados: {stats.get('not_found', 0)}")

            # Mostrar productos sincronizados exitosamente
            if stats.get("created", 0) + stats.get("updated", 0) > 0:
                print(f"\n  ✅ Productos sincronizados en product_catalog_vet:")

                # Query para verificar en BD
                from app.models import ProductCatalogVet

                for code in REAL_VET_CODES:
                    vet_code = f"VET-{code}"
                    product = (
                        db.query(ProductCatalogVet)
                        .filter(ProductCatalogVet.national_code == vet_code)
                        .first()
                    )

                    if product:
                        prescription_status = "🔴 RECETA" if product.cima_requiere_receta else "🟢 LIBRE"
                        print(f"     - {vet_code}: {product.cima_nombre_comercial[:50]}... ({prescription_status})")
                    else:
                        print(f"     - {vet_code}: ❌ NO SINCRONIZADO")

        return result

    except Exception as e:
        print(f"\n❌ Error durante sync: {type(e).__name__}: {str(e)}")
        import traceback

        traceback.print_exc()
        return None

    finally:
        db.close()


async def test_enrich_individual():
    """Prueba de enriquecimiento individual de un producto específico."""
    print("\n" + "=" * 80)
    print("🧪 TEST: Enriquecimiento Individual (enrich_from_cimavet)")
    print("=" * 80)

    test_code = "572240"  # SYNULOX 500/100 mg
    print(f"\n📋 Probando con CN: {test_code}")

    try:
        vet_data = await cimavet_sync_service.enrich_from_cimavet(test_code)

        if vet_data:
            print(f"\n✅ Producto encontrado!")
            print(f"\n  📦 Datos obtenidos:")
            print(f"     - CN: {vet_data.get('codigo_nacional')}")
            print(f"     - Nombre: {vet_data.get('nombre_comercial')[:60]}...")
            print(f"     - Laboratorio: {vet_data.get('laboratorio_titular')}")
            print(f"     - Requiere receta: {vet_data.get('requiere_receta')}")
            print(f"     - Dispensación: {vet_data.get('dispensacion')}")
            print(f"     - PVP: {vet_data.get('pvp')}")
            print(f"     - Principios activos: {len(vet_data.get('principios_activos', []))}")

            if vet_data.get("principios_activos"):
                print(f"\n  💊 Principios activos:")
                for pa in vet_data.get("principios_activos", []):
                    print(f"     - {pa.get('nombre')}")

            return vet_data
        else:
            print(f"\n❌ Producto no encontrado en API CIMAVet")
            return None

    except Exception as e:
        print(f"\n❌ Error: {type(e).__name__}: {str(e)}")
        import traceback

        traceback.print_exc()
        return None


async def main():
    """Ejecuta tests de CIMAVet Sync Incremental."""
    # Test 1: Enriquecimiento individual
    print("\n")
    await test_enrich_individual()

    # Test 2: Sincronización incremental completa
    print("\n")
    result = await test_sync_incremental()

    print("\n" + "=" * 80)
    if result and result.get("status") == "completed":
        print("✅ Tests completados exitosamente")
    else:
        print("⚠️  Tests completados con advertencias/errores")
    print("=" * 80)


if __name__ == "__main__":
    asyncio.run(main())
