"""
Test de integración E2E para CIMAVet enrichment cascade.
Sprint 4 - Issue #354

Verifica:
1. Feature flag CIMAVET_ENABLED activado
2. Cascade: CIMA → Nomenclator → CIMAVet → manual_review
3. Enriquecimiento automático de ventas con productos veterinarios
4. Clasificación correcta RECETA/LIBRE
"""

import asyncio
from datetime import datetime
from sqlalchemy.orm import Session
from app.database import SessionLocal
from app.models import SalesData, ProductCatalog, SalesEnrichment, Pharmacy, User
from app.models.product_catalog_vet import ProductCatalogVet
from app.services.enrichment_service import EnrichmentService
from app.utils.datetime_utils import utc_now
import os

# Productos veterinarios de test (ya sincronizados en Sprint 3)
VET_TEST_PRODUCTS = [
    {"cn": "578403", "name": "SERESTO 4,50 g", "expected_receta": False},  # LIBRE
    {"cn": "572240", "name": "SYNULOX 200/50 mg", "expected_receta": True},  # RECETA
    {"cn": "589610", "name": "APOQUEL 5,4 mg", "expected_receta": True},  # RECETA
]


def test_cimavet_cascade():
    """Test completo del cascade de enriquecimiento CIMAVet."""

    db = SessionLocal()

    try:
        print("=" * 80)
        print("🧪 TEST CIMAVET ENRICHMENT CASCADE")
        print("=" * 80)
        print()

        # 1. Verificar feature flag
        cimavet_enabled = os.getenv("CIMAVET_ENABLED", "false").lower() == "true"
        print(f"1️⃣ Feature Flag CIMAVET_ENABLED: {cimavet_enabled}")

        if not cimavet_enabled:
            print("   ❌ Feature flag desactivado - activar en .env")
            return False
        else:
            print("   ✅ Feature flag activado")
        print()

        # 2. Verificar productos veterinarios en catálogo
        print("2️⃣ Verificando productos veterinarios en catálogo:")
        vet_count = db.query(ProductCatalogVet).count()
        print(f"   Productos en product_catalog_vet: {vet_count}")

        if vet_count == 0:
            print("   ❌ No hay productos veterinarios - ejecutar sync primero")
            return False
        else:
            print("   ✅ Productos disponibles para testing")
        print()

        # 3. Obtener/crear farmacia de desarrollo
        pharmacy = db.query(Pharmacy).filter(Pharmacy.nif == "DEV00000000").first()
        if not pharmacy:
            print("   ⚠️  Creando farmacia de desarrollo...")
            from app.models.pharmacy import Pharmacy as PharmacyModel
            pharmacy = PharmacyModel(
                nif="DEV00000000",
                name="Farmacia Test CIMAVet",
                created_at=utc_now(),
                updated_at=utc_now()
            )
            db.add(pharmacy)
            db.commit()
            db.refresh(pharmacy)
            print(f"   ✅ Farmacia creada: {pharmacy.id}")
        print()

        # 4. Test de enriquecimiento para cada producto
        print("3️⃣ Testing enrichment cascade:")
        print("-" * 80)

        enrichment_service = EnrichmentService()
        results = []

        for test_prod in VET_TEST_PRODUCTS:
            cn = test_prod["cn"]
            name = test_prod["name"]
            expected_receta = test_prod["expected_receta"]

            print(f"\n📦 Producto: {name} (CN: {cn})")
            print(f"   Clasificación esperada: {'RECETA' if expected_receta else 'LIBRE'}")

            # Crear venta simulada
            sale = SalesData(
                pharmacy_id=pharmacy.id,
                upload_id=pharmacy.id,  # Usar pharmacy_id como upload_id temporal
                sale_date=datetime.now(),
                product_name=name,
                codigo_nacional=cn,
                quantity=1,
                total_amount=10.0,
                year=2024,
                month=11,
                day=9,
                created_at=utc_now()
            )
            db.add(sale)
            db.commit()
            db.refresh(sale)

            print(f"   ✅ Venta creada: {sale.id}")

            # Enriquecer venta
            try:
                enrichment = enrichment_service.enrich_single_sale(db, sale)

                if enrichment:
                    # Verificar resultados
                    product = enrichment.product_catalog

                    print(f"   ✅ Enriquecimiento exitoso:")
                    print(f"      - Status: {enrichment.enrichment_status}")
                    print(f"      - Source: {enrichment.enrichment_source}")
                    print(f"      - Prescription category: {enrichment.prescription_category}")
                    print(f"      - Product type: {enrichment.product_type}")

                    if product:
                        print(f"      - Producto: {product.cima_nombre_comercial[:50]}")
                        print(f"      - Uso veterinario: {product.cima_uso_veterinario}")
                        print(f"      - Requiere receta: {product.cima_requiere_receta}")
                        print(f"      - Data sources: {product.data_sources}")

                        # Validar clasificación
                        clasificacion_correcta = (product.cima_requiere_receta == expected_receta)
                        emoji_resultado = "✅" if clasificacion_correcta else "❌"
                        clasificacion_obtenida = "RECETA" if product.cima_requiere_receta else "LIBRE"
                        clasificacion_esperada = "RECETA" if expected_receta else "LIBRE"

                        print(f"      {emoji_resultado} Clasificación: {clasificacion_obtenida} (esperada: {clasificacion_esperada})")

                        results.append({
                            "cn": cn,
                            "success": True,
                            "correct_classification": clasificacion_correcta,
                            "source": enrichment.enrichment_source
                        })
                    else:
                        print(f"      ⚠️  No se encontró producto en catálogo")
                        results.append({"cn": cn, "success": False, "correct_classification": False})
                else:
                    print(f"   ❌ Enriquecimiento falló")
                    results.append({"cn": cn, "success": False, "correct_classification": False})

            except Exception as e:
                print(f"   ❌ Error: {str(e)}")
                results.append({"cn": cn, "success": False, "correct_classification": False, "error": str(e)})

            # Limpiar venta de test
            db.delete(sale)
            db.commit()

        # 5. Resumen final
        print()
        print("=" * 80)
        print("📊 RESUMEN DE RESULTADOS:")
        print("=" * 80)

        total = len(results)
        exitosos = sum(1 for r in results if r.get("success"))
        clasificacion_correcta = sum(1 for r in results if r.get("correct_classification"))

        print(f"Total productos testeados: {total}")
        print(f"Enriquecimientos exitosos: {exitosos}/{total}")
        print(f"Clasificaciones correctas: {clasificacion_correcta}/{total}")

        if exitosos == total and clasificacion_correcta == total:
            print()
            print("✅ ¡TEST EXITOSO! CIMAVet cascade funcionando correctamente")
            return True
        else:
            print()
            print("❌ Test falló - revisar cascade")
            return False

    except Exception as e:
        print(f"\n❌ Error en test: {str(e)}")
        import traceback
        traceback.print_exc()
        return False

    finally:
        db.close()


if __name__ == "__main__":
    success = test_cimavet_cascade()
    exit(0 if success else 1)
