#!/usr/bin/env python3
"""
Script de prueba para verificar conectividad con API CIMAVet.
Issue #354 - Verificación de integración CIMAVet.
"""

import asyncio
import sys
from pathlib import Path

# Add backend to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.services.cimavet_sync_service import cimavet_sync_service
from app.database import SessionLocal
import structlog

logger = structlog.get_logger(__name__)


async def test_cimavet_api():
    """Prueba de conectividad con API CIMAVet."""
    print("=" * 80)
    print("🧪 TEST: Conectividad API CIMAVet")
    print("=" * 80)

    # 1. Verificar configuración del servicio
    print(f"\n📋 Configuración:")
    print(f"  - API URL: {cimavet_sync_service.api_base_url}")
    print(f"  - Timeout: {cimavet_sync_service.timeout.read}s")
    print(f"  - Sync interval: {cimavet_sync_service.sync_interval_days} días")

    # 2. Intentar llamar a la API
    print(f"\n🌐 Intentando conectar a API CIMAVet...")
    try:
        vet_products = await cimavet_sync_service._fetch_cimavet_api()

        if vet_products:
            print(f"✅ API CIMAVet respondió exitosamente!")
            print(f"  - Productos recibidos: {len(vet_products)}")

            # Mostrar primeros 3 productos como muestra
            if len(vet_products) > 0:
                print(f"\n📦 Muestra de productos (primeros 3):")
                for i, prod in enumerate(vet_products[:3], 1):
                    print(f"\n  Producto {i}:")
                    print(f"    - CN: {prod.get('codigo_nacional')}")
                    print(f"    - Nombre: {prod.get('nombre_comercial')[:60]}...")
                    print(f"    - Requiere receta: {prod.get('requiere_receta')}")
                    print(f"    - Laboratorio: {prod.get('laboratorio_titular')}")

            return True, len(vet_products)
        else:
            print(f"⚠️  API respondió pero sin productos")
            return False, 0

    except Exception as e:
        print(f"❌ Error al conectar con API CIMAVet:")
        print(f"   {type(e).__name__}: {str(e)}")
        return False, 0


async def test_sync(force=False):
    """Prueba de sincronización completa."""
    print("\n" + "=" * 80)
    print("🔄 TEST: Sincronización CIMAVet")
    print("=" * 80)

    db = SessionLocal()
    try:
        result = await cimavet_sync_service.sync_cimavet_catalog(db, force=force)

        print(f"\n📊 Resultado de sincronización:")
        print(f"  - Status: {result.get('status')}")

        if result.get('stats'):
            stats = result['stats']
            print(f"  - Procesados: {stats.get('processed', 0)}")
            print(f"  - Creados: {stats.get('created', 0)}")
            print(f"  - Actualizados: {stats.get('updated', 0)}")
            print(f"  - Errores: {stats.get('errors', 0)}")

        return result

    finally:
        db.close()


async def main():
    """Ejecuta tests de CIMAVet."""
    # Test 1: Conectividad API
    api_ok, count = await test_cimavet_api()

    if not api_ok:
        print(f"\n❌ No se pudo conectar a la API CIMAVet")
        print(f"   Verifica la URL en la documentación AEMPS")
        return

    # Test 2: Sincronización (solo si API funciona)
    print(f"\n¿Ejecutar sincronización completa? (y/n): ", end="")
    response = input().strip().lower()

    if response == 'y':
        await test_sync(force=True)
    else:
        print(f"\n✅ Test de conectividad completado. Saltando sincronización.")

    print("\n" + "=" * 80)
    print("✅ Tests completados")
    print("=" * 80)


if __name__ == "__main__":
    asyncio.run(main())
