#!/usr/bin/env python3
"""
Script para sincronizar productos de prescription_reference_list a ProductCatalog.

Issue #449: Los productos de prescription_reference_list (como tiras reactivas)
necesitan tener entradas en ProductCatalog para aparecer en analytics.

Este script:
1. Lee todos los productos de prescription_reference_list
2. Para cada producto que NO esté en ProductCatalog, crea una entrada
3. Actualiza SalesEnrichment para vincular product_catalog_id

Uso:
    docker-compose exec backend python scripts/sync_reference_to_catalog.py
    docker-compose exec backend python scripts/sync_reference_to_catalog.py --dry-run
"""
import argparse
import sys
from pathlib import Path

# Añadir raíz del proyecto al PYTHONPATH
sys.path.insert(0, str(Path(__file__).parent.parent))

from sqlalchemy import and_, func
from app.database import SessionLocal
from app.models.enums import PrescriptionCategory
from app.models.prescription_reference_list import PrescriptionReferenceList
from app.models.product_catalog import ProductCatalog
from app.models.sales_enrichment import SalesEnrichment


def main():
    parser = argparse.ArgumentParser(
        description="Sincronizar prescription_reference_list a ProductCatalog"
    )
    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="Solo mostrar qué se haría, sin hacer cambios"
    )
    args = parser.parse_args()

    print("=" * 80)
    print("SINCRONIZACIÓN prescription_reference_list → ProductCatalog")
    print("=" * 80)
    if args.dry_run:
        print("[DRY-RUN] No se realizarán cambios\n")
    else:
        print()

    session = SessionLocal()

    try:
        # 1. Obtener productos de reference_list que NO están en ProductCatalog
        print("[1/3] Buscando productos sin entrada en ProductCatalog...")

        # Subconsulta de códigos nacionales en ProductCatalog
        existing_codes = session.query(ProductCatalog.national_code).subquery()

        # Productos en reference_list que NO están en ProductCatalog
        missing = (
            session.query(PrescriptionReferenceList)
            .filter(~PrescriptionReferenceList.national_code.in_(existing_codes))
            .all()
        )

        print(f"      → {len(missing)} productos a crear en ProductCatalog")

        if not missing:
            print("\n[INFO] Todos los productos ya tienen entrada en ProductCatalog")
            return

        # Desglose por categoría
        by_category = {}
        for ref in missing:
            cat = ref.category
            by_category[cat] = by_category.get(cat, 0) + 1

        print("\n      Desglose por categoría:")
        for cat, count in sorted(by_category.items()):
            print(f"        - {cat}: {count}")

        # 2. Crear entradas en ProductCatalog
        print(f"\n[2/3] Creando entradas en ProductCatalog...")

        created_products = {}
        for ref in missing:
            # Convertir category string a enum si es necesario
            if isinstance(ref.category, str):
                try:
                    category_enum = PrescriptionCategory(ref.category)
                except ValueError:
                    print(f"      [WARN] Categoría inválida: {ref.category}")
                    continue
            else:
                category_enum = ref.category

            if not args.dry_run:
                product = ProductCatalog(
                    national_code=ref.national_code,
                    cima_nombre_comercial=ref.product_name,
                    nomen_nombre=ref.product_name,
                    xfarma_prescription_category=category_enum,
                    data_sources=f"prescription_reference_list:{ref.reference_source}",
                )
                session.add(product)
                session.flush()  # Obtener ID
                created_products[ref.national_code] = product.id

        if not args.dry_run:
            print(f"      → {len(created_products)} productos creados")
        else:
            print(f"      → {len(missing)} productos se crearían")

        # 3. Actualizar SalesEnrichment con product_catalog_id
        print(f"\n[3/3] Actualizando SalesEnrichment...")

        if not args.dry_run:
            # Para cada producto creado, buscar SalesEnrichment con ese código nacional
            # y actualizar product_catalog_id
            updated_count = 0

            for national_code, product_id in created_products.items():
                # Buscar SalesEnrichment por código nacional
                result = (
                    session.query(SalesEnrichment)
                    .join(SalesEnrichment.sales_data)
                    .filter(
                        and_(
                            SalesEnrichment.product_catalog_id.is_(None),
                            SalesEnrichment.sales_data.has(codigo_nacional=national_code)
                        )
                    )
                    .update(
                        {SalesEnrichment.product_catalog_id: product_id},
                        synchronize_session=False
                    )
                )
                updated_count += result

            print(f"      → {updated_count} registros de SalesEnrichment actualizados")

            # Commit
            session.commit()
        else:
            # Contar cuántos se actualizarían
            count = (
                session.query(func.count(SalesEnrichment.id))
                .join(SalesEnrichment.sales_data)
                .filter(
                    and_(
                        SalesEnrichment.product_catalog_id.is_(None),
                        SalesEnrichment.sales_data.has(
                            codigo_nacional.in_([ref.national_code for ref in missing])
                        )
                    )
                )
            ).scalar()
            print(f"      → ~{count} registros se actualizarían")

        # Resumen
        print(f"\n{'=' * 80}")
        print("SINCRONIZACIÓN COMPLETADA" if not args.dry_run else "[DRY-RUN] Simulación completada")
        print(f"{'=' * 80}")

    except Exception as e:
        session.rollback()
        print(f"\n[ERROR] {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)
    finally:
        session.close()


if __name__ == "__main__":
    main()
