"""
Spike: Exploración de encargos en Farmanager
=============================================

Este script "radiografía" la base de datos MySQL de Farmanager para
descubrir dónde se almacenan los encargos de clientes.

Ejecutar:
    python backend/scripts/spikes/farmanager_encargos_spike.py

Requisitos:
    pip install sqlalchemy pymysql pandas

Seguridad:
    100% READ-ONLY (solo SELECT)

Output:
    Documento para docs/FARMANAGER_ENCARGOS_SCHEMA.md
"""

import sys
import os
from datetime import datetime
from pathlib import Path

# Añadir backend al path para imports
sys.path.insert(0, str(Path(__file__).parent.parent.parent))

import pandas as pd
from sqlalchemy import create_engine, text

# CONFIGURACION - FARMANAGER USA MYSQL
# Credenciales de backend/app/erp_adapters/farmanager/test_connection.py
DB_HOST = "localhost"
DB_PORT = 3306
DB_NAME = "gesql"  # Nombre real de la base de datos Farmanager
DB_USER = "administrador"
DB_PASS = "nimda2009"
DB_URL = f"mysql+pymysql://{DB_USER}:{DB_PASS}@{DB_HOST}:{DB_PORT}/{DB_NAME}?charset=utf8mb4"

# Archivo de output para documentacion
OUTPUT_FILE = Path(__file__).parent.parent.parent.parent / "docs" / "FARMANAGER_ENCARGOS_SCHEMA.md"


def print_header(title: str) -> None:
    print(f"\n{'='*60}\n {title}\n{'='*60}")


def safe_query(conn, query: str, description: str) -> pd.DataFrame | None:
    """Ejecuta query de forma segura, retorna DataFrame o None si falla"""
    try:
        return pd.read_sql(query, conn)
    except Exception as e:
        print(f"  [!] Error en {description}: {e}")
        return None


def run_spike():
    """
    Detective de tablas: Busca donde se esconden los encargos en Farmanager.
    """
    print(f"{'='*60}")
    print(f" SPIKE DE ENCARGOS FARMANAGER")
    print(f" Fecha: {datetime.now().strftime('%Y-%m-%d %H:%M')}")
    print(f" DB: {DB_NAME}@{DB_HOST}:{DB_PORT}")
    print(f"{'='*60}")

    # Conectar
    try:
        engine = create_engine(DB_URL)
        conn = engine.connect()
        print("\n[OK] Conexion establecida")
    except Exception as e:
        print(f"\n[X] ERROR CRITICO DE CONEXION: {e}")
        print("Tip: Verifica que MySQL esta corriendo y las credenciales son correctas.")
        return

    results = {}  # Para documentacion

    # 1. BUSQUEDA DE TABLAS SOSPECHOSAS POR NOMBRE
    print_header("1. TABLAS CANDIDATAS (Por nombre)")
    query = f"""
    SELECT TABLE_NAME as TABLA
    FROM INFORMATION_SCHEMA.TABLES
    WHERE TABLE_SCHEMA = '{DB_NAME}'
    AND (TABLE_NAME LIKE '%encarg%'
      OR TABLE_NAME LIKE '%reserv%'
      OR TABLE_NAME LIKE '%aparta%'
      OR TABLE_NAME LIKE '%pend%'
      OR TABLE_NAME LIKE '%ges4%')
    ORDER BY 1
    """
    df = safe_query(conn, query, "busqueda tablas encargos")
    if df is not None and not df.empty:
        print(df.to_string(index=False))
        results['tablas_encargos'] = df['TABLA'].tolist()
    else:
        print("  [!] Ninguna tabla encontrada con patrones ENCARG/RESERV/APARTA/PEND")
        results['tablas_encargos'] = []

    # 2. BUSQUEDA PROFUNDA DE COLUMNAS SOSPECHOSAS
    print_header("2. COLUMNAS SOSPECHOSAS (En toda la BD)")
    query = f"""
    SELECT TABLE_NAME as TABLA, COLUMN_NAME as CAMPO
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = '{DB_NAME}'
    AND (COLUMN_NAME LIKE '%encarg%'
      OR COLUMN_NAME LIKE '%reserv%'
      OR COLUMN_NAME LIKE '%aparta%'
      OR COLUMN_NAME LIKE '%estado%'
      OR COLUMN_NAME LIKE '%pend%')
    ORDER BY 1, 2
    """
    df = safe_query(conn, query, "busqueda columnas")
    if df is not None and not df.empty:
        print(df.head(30).to_string(index=False))
        results['columnas_sospechosas'] = df.to_dict('records')
    else:
        print("  [!] Sin columnas sospechosas")

    # 3. ANALISIS DE VOLUMEN (Row Counts)
    print_header("3. VOLUMEN DE DATOS")
    targets = ['ges404', 'ges406', 'ges050', 'ges051', 'ges301', 'ges302', 'ges407']
    results['volumenes'] = {}
    for t in targets:
        try:
            count = pd.read_sql(f"SELECT COUNT(*) as CNT FROM {t}", conn).iloc[0, 0]
            print(f"  {t.ljust(10)}: {count:>10,} registros")
            results['volumenes'][t] = count
        except:
            print(f"  {t.ljust(10)}: No existe o inaccesible")
            results['volumenes'][t] = None

    # 4. INSPECCION DE GES404 (Propuestas/Pedidos)
    print_header("4. INSPECCION ges404 (Propuestas de Pedido)")
    df = safe_query(conn, "SELECT * FROM ges404 ORDER BY 1 DESC LIMIT 5", "ges404 sample")
    if df is not None:
        print(f"  Columnas: {list(df.columns)}")
        print(f"\n  Muestra (5 filas):")
        print(df.to_string(index=False))
        results['ges404_columns'] = list(df.columns)

    # 5. INSPECCION DE GES050 (Clientes)
    print_header("5. INSPECCION ges050 (Clientes)")
    df = safe_query(conn, "SELECT * FROM ges050 LIMIT 3", "ges050 sample")
    if df is not None:
        # Filtrar columnas que puedan ser de encargos
        encargo_cols = [c for c in df.columns if any(x in c.upper() for x in ['ENCARG', 'RESERV', 'PEND', 'DEBE'])]
        print(f"  Columnas totales: {len(df.columns)}")
        print(f"  Columnas sospechosas: {encargo_cols if encargo_cols else 'Ninguna'}")
        results['ges050_encargo_cols'] = encargo_cols

    # 6. HIPOTESIS: VENTAS EN ESPERA (ges301 con estado especial)
    print_header("6. HIPOTESIS: Ventas en estado especial")
    # Buscar si hay columna ESTADO o similar
    df = safe_query(conn, "SELECT * FROM ges301 LIMIT 1", "ges301 schema")
    if df is not None:
        estado_cols = [c for c in df.columns if 'ESTADO' in c.upper() or 'TIPO' in c.upper()]
        print(f"  Columnas de estado en ges301: {estado_cols}")

        if estado_cols:
            for col in estado_cols[:2]:  # Max 2 columnas
                df_estados = safe_query(
                    conn,
                    f"SELECT {col}, COUNT(*) as CNT FROM ges301 GROUP BY {col} ORDER BY 2 DESC",
                    f"ges301 {col}"
                )
                if df_estados is not None:
                    print(f"\n  Valores de {col}:")
                    print(df_estados.to_string(index=False))

    # 7. BUSCAR TABLA ESPECIFICA DE ENCARGOS
    print_header("7. BUSQUEDA DIRECTA: Todas las tablas GES")
    # Listar todas las tablas que empiezan con GES
    query = f"""
    SELECT TABLE_NAME as TABLA
    FROM INFORMATION_SCHEMA.TABLES
    WHERE TABLE_SCHEMA = '{DB_NAME}'
    AND TABLE_NAME LIKE 'ges%'
    ORDER BY 1
    """
    df = safe_query(conn, query, "listar tablas GES")
    if df is not None:
        print(f"  Tablas GES encontradas: {len(df)}")
        print(df.to_string(index=False))
        ges_tables = df['TABLA'].tolist()
        results['ges_tables'] = ges_tables

    # 8. INSPECCION DE GES406 (Lineas de pedido)
    print_header("8. INSPECCION ges406 (Lineas de pedido)")
    df = safe_query(conn, "SELECT * FROM ges406 ORDER BY 1 DESC LIMIT 5", "ges406 sample")
    if df is not None:
        print(f"  Columnas: {list(df.columns)}")
        # Buscar columna de cliente
        cli_cols = [c for c in df.columns if 'CLI' in c.upper() or 'CLIENTE' in c.upper()]
        print(f"  Columnas de cliente: {cli_cols}")
        results['ges406_columns'] = list(df.columns)

    # CERRAR CONEXION
    conn.close()

    # GENERAR RESUMEN
    print_header("RESUMEN DEL SPIKE")
    print(f"""
Tablas encontradas con patron ENCARG/RESERV: {len(results.get('tablas_encargos', []))}
Volumenes:
  - ges404 (propuestas): {results['volumenes'].get('ges404', 'N/A')}
  - ges406 (lineas): {results['volumenes'].get('ges406', 'N/A')}
  - ges050 (clientes): {results['volumenes'].get('ges050', 'N/A')}

SIGUIENTE PASO:
  1. Revisar output y identificar tabla/columna de encargos
  2. Documentar en docs/FARMANAGER_ENCARGOS_SCHEMA.md
  3. Implementar query get_customer_encargos() en adapter
""")

    print(f"\n[OK] SPIKE FINALIZADO - {datetime.now().strftime('%H:%M:%S')}")


if __name__ == "__main__":
    run_spike()
