#!/usr/bin/env python3
"""
Script para resetear la base de datos en Render eliminando todas las tablas
y volviendo a ejecutar las migraciones desde cero.

Uso en Render (via shell):
    python backend/scripts/reset_database_render.py
"""

import os
import sys
from pathlib import Path

# Añadir el directorio raíz al path
sys.path.insert(0, str(Path(__file__).parent.parent.parent))

from sqlalchemy import create_engine, inspect, text


def drop_all_tables():
    """Elimina todas las tablas de la base de datos."""

    database_url = os.getenv("DATABASE_URL")
    if not database_url:
        raise ValueError("DATABASE_URL environment variable not set")

    # Render usa postgres:// pero SQLAlchemy 1.4+ requiere postgresql://
    if database_url.startswith("postgres://"):
        database_url = database_url.replace("postgres://", "postgresql://", 1)

    engine = create_engine(database_url)

    print("🔍 Conectando a la base de datos...")

    with engine.connect() as conn:
        # Obtener lista de tablas
        inspector = inspect(engine)
        tables = inspector.get_table_names()

        if not tables:
            print("✅ La base de datos ya está vacía")
            return

        print(f"📋 Tablas encontradas: {len(tables)}")
        for table in tables:
            print(f"   - {table}")

        # Eliminar todas las tablas en cascada
        print("\n🗑️  Eliminando todas las tablas...")

        # Usar CASCADE para eliminar tablas con dependencias
        for table in tables:
            try:
                conn.execute(text(f'DROP TABLE IF EXISTS "{table}" CASCADE'))
                print(f"   ✓ Eliminada: {table}")
            except Exception as e:
                print(f"   ✗ Error eliminando {table}: {e}")

        conn.commit()
        print("\n✅ Todas las tablas eliminadas exitosamente")


def run_migrations():
    """Ejecuta las migraciones de Alembic."""
    import subprocess

    print("\n🔄 Ejecutando migraciones de Alembic...")

    # Cambiar al directorio backend
    backend_dir = Path(__file__).parent.parent

    try:
        result = subprocess.run(
            ["alembic", "upgrade", "head"], cwd=backend_dir, capture_output=True, text=True, check=True
        )

        print(result.stdout)

        if result.stderr:
            print("Advertencias:", result.stderr)

        print("✅ Migraciones ejecutadas exitosamente")

    except subprocess.CalledProcessError as e:
        print(f"❌ Error ejecutando migraciones: {e}")
        print(f"STDOUT: {e.stdout}")
        print(f"STDERR: {e.stderr}")
        sys.exit(1)


def main():
    """Función principal."""

    print("=" * 60)
    print("🔧 RESET DE BASE DE DATOS - Render Production")
    print("=" * 60)
    print()

    # Verificar que estamos en Render
    is_render = os.getenv("RENDER") == "true"
    if not is_render:
        response = input("⚠️  No estás en Render. ¿Continuar? (yes/no): ")
        if response.lower() != "yes":
            print("Operación cancelada")
            sys.exit(0)

    # Confirmar acción
    print("⚠️  ADVERTENCIA: Esta operación eliminará TODAS las tablas de la base de datos")
    print("   y volverá a ejecutar las migraciones desde cero.")
    print()

    if is_render:
        # En Render, ejecutar automáticamente
        print("✓ Ejecutando en Render - procediendo automáticamente...")
    else:
        response = input("¿Estás seguro? Escribe 'DELETE ALL TABLES' para confirmar: ")
        if response != "DELETE ALL TABLES":
            print("Operación cancelada")
            sys.exit(0)

    try:
        # Paso 1: Eliminar todas las tablas
        drop_all_tables()

        # Paso 2: Ejecutar migraciones
        run_migrations()

        print()
        print("=" * 60)
        print("✅ Base de datos reseteada exitosamente")
        print("=" * 60)

    except Exception as e:
        print()
        print("=" * 60)
        print(f"❌ Error durante el reset: {e}")
        print("=" * 60)
        sys.exit(1)


if __name__ == "__main__":
    main()
