#!/usr/bin/env python3
"""
Script para re-enriquecer ventas con product_type legacy.

Actualiza:
- 'otc' → 'free_sale'
- 'parapharmacy' → 'free_sale'

Aplica nueva lógica basada en xfarma_prescription_category.

Uso:
    python scripts/reenrich_legacy_types.py
    python scripts/reenrich_legacy_types.py --dry-run
"""
import argparse
import sys
import logging
from pathlib import Path
from datetime import datetime, timezone

# Añadir raíz del proyecto al PYTHONPATH
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.database import SessionLocal
from app.models.sales_enrichment import SalesEnrichment
from app.models.product_catalog import ProductCatalog
from app.services.enrichment_service import EnrichmentService

# Configurar logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)


def reenrich_legacy_records(db, dry_run=False):
    """Re-enriquece registros con product_type legacy"""

    # Contar registros legacy
    legacy_count = db.query(SalesEnrichment).filter(
        SalesEnrichment.product_type.in_(['otc', 'parapharmacy'])
    ).count()

    logger.info(f"Registros legacy encontrados: {legacy_count:,}")

    if dry_run:
        logger.info("[DRY-RUN] No se actualizarán registros")
        return

    # Inicializar servicio de enriquecimiento
    enrichment_service = EnrichmentService(db)

    # Procesar en batches
    batch_size = 1000
    updated_count = 0
    errors = 0

    logger.info("Iniciando re-enriquecimiento...")

    while True:
        # Obtener batch de registros legacy
        batch = db.query(SalesEnrichment).filter(
            SalesEnrichment.product_type.in_(['otc', 'parapharmacy'])
        ).limit(batch_size).all()

        if not batch:
            break

        for enrichment in batch:
            try:
                # Obtener producto del catálogo
                product_catalog = db.query(ProductCatalog).filter(
                    ProductCatalog.id == enrichment.product_catalog_id
                ).first()

                if not product_catalog:
                    logger.warning(f"Producto no encontrado: {enrichment.product_catalog_id}")
                    # Actualizar a free_sale directamente
                    enrichment.product_type = 'free_sale'
                    continue

                # Re-derivar product_type con nueva lógica
                new_product_type = enrichment_service._derive_product_type(product_catalog)
                old_product_type = enrichment.product_type

                # Actualizar
                enrichment.product_type = new_product_type

                # Log si cambió
                if old_product_type != new_product_type:
                    logger.debug(
                        f"Actualizado: {enrichment.id} | "
                        f"{old_product_type} → {new_product_type} | "
                        f"CN: {product_catalog.national_code}"
                    )

                updated_count += 1

                if updated_count % 100 == 0:
                    logger.info(f"Progreso: {updated_count:,}/{legacy_count:,} registros actualizados")
                    db.commit()  # Commit incremental

            except Exception as e:
                logger.error(f"Error procesando {enrichment.id}: {e}")
                errors += 1
                continue

        # Commit del batch
        db.commit()

    logger.info(f"Re-enriquecimiento completado: {updated_count:,} registros actualizados")
    if errors > 0:
        logger.warning(f"Errores: {errors}")

    return updated_count, errors


def verify_cleanup(db):
    """Verifica que no quedan registros legacy"""
    legacy_count = db.query(SalesEnrichment).filter(
        SalesEnrichment.product_type.in_(['otc', 'parapharmacy'])
    ).count()

    if legacy_count == 0:
        logger.info("✅ Verificación exitosa: 0 registros legacy restantes")
    else:
        logger.warning(f"⚠️  Todavía quedan {legacy_count:,} registros legacy")

    # Mostrar distribución actualizada
    distribution = db.query(
        SalesEnrichment.product_type,
        db.func.count(SalesEnrichment.id)
    ).group_by(
        SalesEnrichment.product_type
    ).all()

    logger.info("\nDistribución actualizada de product_type:")
    for ptype, count in sorted(distribution, key=lambda x: x[1], reverse=True):
        display_type = ptype if ptype else "NULL"
        logger.info(f"  {display_type:.<30} {count:>8,}")

    return legacy_count == 0


def main():
    parser = argparse.ArgumentParser(
        description="Re-enriquecer ventas con product_type legacy"
    )
    parser.add_argument(
        '--dry-run',
        action='store_true',
        help="Solo mostrar estadísticas sin actualizar"
    )
    args = parser.parse_args()

    print("=" * 80)
    print("RE-ENRIQUECIMIENTO DE VENTAS (product_type legacy)")
    print("=" * 80)
    print(f"Timestamp: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    print(f"Modo: {'DRY-RUN' if args.dry_run else 'ACTUALIZACIÓN REAL'}")
    print("=" * 80)
    print()

    db = SessionLocal()

    try:
        # Re-enriquecer
        updated, errors = reenrich_legacy_records(db, args.dry_run)

        if not args.dry_run:
            print()
            print("=" * 80)
            print("VERIFICACIÓN POST-ACTUALIZACIÓN")
            print("=" * 80)

            # Verificar limpieza
            is_clean = verify_cleanup(db)

            print()
            print("=" * 80)
            print("RESUMEN")
            print("=" * 80)
            print(f"  Registros actualizados: {updated:,}")
            print(f"  Errores: {errors}")
            print(f"  Estado: {'✅ LIMPIO' if is_clean else '⚠️  REVISAR'}")
            print("=" * 80)

    except Exception as e:
        logger.error(f"Error durante re-enriquecimiento: {e}", exc_info=True)
        db.rollback()
        sys.exit(1)
    finally:
        db.close()


if __name__ == "__main__":
    main()
