#!/usr/bin/env python
"""
Script para procesar la lista de 500 marcas del usuario.
Extrae marcas únicas, filtra las existentes, y genera el código para añadir.

Issue #446: Taxonomía jerárquica venta libre - Fase 2 Brand Expansion
"""

import re
import sys

sys.path.insert(0, "/app")

# Lista completa de marcas del usuario (500 marcas en 50 categorías)
USER_BRANDS = {
    # DERMOCOSMÉTICA FACIAL - Premium/Selectiva
    "proteccion_solar": [
        "La Roche-Posay", "Vichy", "Avène", "Bioderma", "Eucerin", "CeraVe", "SVR",
        "Ducray", "Lierac", "Filorga", "Caudalie", "Nuxe", "Galenic", "Esthederm",
        "Medik8", "SkinCeuticals", "Neostrata", "Endocare", "Martiderm", "Sesderma",
        "ISDIN", "Sensilis", "Be+", "Rilastil", "Cumlaude Lab", "Isdinceutics",
        "Two Poles", "5punto5",
    ],
    "arrugas_antiedad": [
        "Arturo Alba", "GH", "Fridda Dorsch", "Olyan Farma", "Nezeni", "Bella Aurora",
        "MartiDerm", "Massada", "Deliplus", "Babaria", "Instituto Español", "Leti",
        "Xhekpon", "Th Pharma", "D'Shila", "Singuladerm", "Ozoaqua",
        "Retincare", "Flavo-C", "Redermic", "Hyalu B5", "Liftactiv", "Neovadiol",
        "Toleriane", "Pigmentclar", "Effaclar", "Sebium", "Sensibio", "Atoderm",
        "Cicabio", "Cicaplast", "Epithéliale", "Cicalfate", "A-Derma", "Klorane",
        "Phyto", "Uriage", "Embryolisse", "Topicrem", "Noreva", "ACM", "Isis Pharma",
        "Talika", "Carita", "Anne Möller", "Lancaster", "Collistar",
    ],
    # PROTECCIÓN SOLAR
    "proteccion_solar_2": [
        "Heliocare", "ISDIN Fotoprotector", "La Roche-Posay Anthelios", "Avène Solar",
        "Eucerin Sun", "Vichy Capital Soleil", "Bioderma Photoderm", "Ladival",
        "Piz Buin", "Hawaiian Tropic", "Nivea Sun", "Garnier Delial", "Rilastil Sun",
        "Sensilis Sun", "Be+ Skin Protect", "Mustela Solar", "Apivita Suncare",
        "Caudalie Solaire", "Nuxe Sun", "Lancaster Sun", "Shiseido Sun", "Clarins Sun",
        "Biotherm Solaire", "Clinique Sun", "Neutrogena Sun", "Coppertone",
        "Australian Gold", "Ecran", "Babaria Solar", "Delial",
    ],
    # CAPILAR / ANTICAÍDA
    "caida_cabello": [
        "Olistic", "Pilexil", "Iraltone", "Complidermol", "Lambdapil", "Klorane",
        "René Furterer", "Phyto", "Ducray", "Vichy Dercos", "Alpecin", "Minoxidil Viñas",
        "Regaxidil", "Lacer Pilopeptan", "Isdin Lambdapil", "Nuggela & Sulé", "Revita",
        "Nanogen", "Nioxin", "Kerastase", "L'Oréal Serioxyl", "Redken Cerafill",
        "Aminexil", "Neogenic", "Densiage", "Anacaps", "Keratincaps", "Vitalfan",
        "Triphasic", "Forticea", "Lazartigue", "Moncho Moreno", "Valquer", "Kerzo",
        "Svenson", "NHCO Aminohair", "Oenobiol Capilar", "Arkocapil", "Innéov",
        "Dercos Aminexil",
    ],
    # DOLOR / ANALGÉSICOS OTC
    "dolor": [
        "Nurofen", "Ibuprofeno", "Paracetamol", "Gelocatil", "Apiretal", "Dalsy",
        "Espididol", "Ibudol", "Ibufen", "Dolostop", "Antidol", "Aspirina",
        "Efferalgan", "Termalgin", "Dolocatil", "Actron", "Neobrufen", "Difenadol",
        "Difenatil", "Endolex", "Dolovanz", "Geloprofen", "Dexketoprofeno",
        "Enantyum", "Adolonta", "Voltadol", "Flogoprofen", "Diclofenaco tópico",
        "Reflex", "Fisiocrem", "Radio Salil", "Thrombocid", "Hirudoid", "Arnidol",
        "Calmazine",
    ],
    # DOLOR MUSCULAR / TÓPICOS
    "dolor_muscular": [
        "Voltadol", "Flogoprofen", "Diclokern", "Reflex", "Fisiocrem", "Radio Salil",
        "Calmatel", "Fastum", "Feldene", "Voltaren Emulgel", "Tiger Balm",
        "Bálsamo de Tigre", "Counterpain", "Bengué", "Calmazine", "Zenavan",
        "Cinfadol", "Algesal", "Traumeel", "Kytta", "Dolalgial", "Thrombocid",
        "Hirudoid", "Menaven", "Hidrotelial Luxoben",
    ],
    # GRIPE Y RESFRIADO
    "gripe_resfriado": [
        "Frenadol", "Pharmagrip", "Couldina", "Ilvico", "Grippal", "Normogrip",
        "Bisolgrip", "Propalgina", "Termalgin Gripe", "Gelocatil Gripe",
        "Aspirina Complex", "Vincigrip", "Actigrip", "Antigripine", "Respibien",
        "Utabon", "Rhinospray", "Nasalvicks", "Vicks VapoRub", "Inhalvicks",
        "Respidina", "Rhinomer", "Physiomer", "Sterimar", "Marimer", "Rhinodouche",
        "Flulenza", "Normonasal", "Otrivina", "Afrin",
    ],
    # TOS Y MUCOSIDAD
    "tos_mucosidad": [
        "Fluimucil", "Mucosan", "Flutox", "Iniston", "Bisolvon", "Prospantus",
        "Cinfatos", "Cinfamucol", "Mucibron", "Frionex", "Pectox", "Romilar",
        "Toseina", "Codeisan", "Cloperastina", "Ambroxol", "Acetilcisteína",
        "Carbocisteína", "Bromhexina", "Grintuss", "Aboca", "Propolaid", "Stodal",
        "Drosetux", "Seditus", "Bronchostop", "Normotus", "Zarbees",
        "Sante Verte Sediflu", "Juanola",
    ],
    # GARGANTA
    "dolor_garganta": [
        "Strepsils", "Bucomax", "Lizipaina", "Angileptol", "Strefen", "Faringesic",
        "Garydol", "Propalgar", "Juanolgar", "Pharysol", "Tantum Verde", "Angal",
        "Bucometasana", "Drill", "Mebucaína", "Lysopaïne", "Faringospray",
        "Anginovag", "Deratin", "Propoleo", "Própolis", "Ricola", "Vivis", "Praims",
        "Sediflu Garganta",
    ],
    # DIGESTIVO - ACIDEZ/REFLUJO
    "acidez_reflujo": [
        "Almax", "Gaviscon", "Rennie", "Omeprazol", "Omereflux", "Omekaste",
        "Pantoprazol", "Lansoprazol", "Maalox", "Mylanta", "Phosphalugel",
        "Iberogast", "Motilium", "Primperan", "Esoxx", "Arcid", "Esophacare",
        "Neobianacid", "Oliovita Gastro", "Acidrine", "Gavidigest", "Digest",
        "ENO", "Sal de Fruta", "Alka-Seltzer",
    ],
    # DIGESTIVO - GASES
    "gases_digestion": [
        "Aero Red", "Benegast", "Fuca", "Aquilea Gases", "NS Digestconfort",
        "Naturflat", "Digest", "Carbón vegetal", "Simethicone", "Dimeticona",
        "Nutira", "Fructaid", "Lactasa", "Lactoben", "Digerflat",
    ],
    # DIGESTIVO - ESTREÑIMIENTO
    "estrenimiento": [
        "Dulcolaxo", "Micralax", "Fisioenema", "Melilax", "Plantaben", "Duphalac",
        "Movicol", "Dulcosoft", "Evacuol", "Bekunis", "Aliviolas", "Fave de Fuca",
        "Roha Max", "Emportal", "Linomed", "Supositorios glicerina", "Glycilax",
        "Laxante Salud", "Laxante Bescansa", "Vilardell Digest", "Casenlax",
        "Macrogol", "Lactulosa", "Regulax", "Agiolax",
    ],
    # DIGESTIVO - DIARREA
    "diarrea": [
        "Fortasec", "Salvacolina", "Sindiar", "Tiorfan", "Ultra-Levura", "Sueroral",
        "Suero Oral Normon", "Electrolit", "Bioralsuero", "Megasmect", "Tanagel",
        "Smecta", "Dialiv", "NS Suerobiotic", "Probiobal",
    ],
    # PROBIÓTICOS / FLORA INTESTINAL
    "flora_intestinal": [
        "Ultra-Levura", "Lactoflora", "Lactibiane", "Bivos", "Reuteri", "Megalevure",
        "Rotagermine", "Florabiotic", "Prodefen", "Vivomixx", "VSL#3", "Alflorex",
        "Kijimea", "Yovis", "Profaes4", "Produo", "Optibac", "Solgar", "Lamberts",
        "Solaray", "Vitanatur", "Aquilea Probio", "Arkoprobiotics", "Supradyn Probio",
        "Multibionta", "Pearls", "Florastor", "Enterogermina", "Bioithas", "Sibok",
    ],
    # VITAMINAS Y SUPLEMENTOS GENERALES
    "vitaminas_general": [
        "Supradyn", "Multicentrum", "Pharmaton", "Berocca", "Redoxon", "Dayamineral",
        "Hidropolivit", "Becozyme", "Hidroxil", "Vitagobens", "Medebiotin", "Aquilea",
        "NS Vitans", "Arkopharma", "Solgar", "Solaray", "Lamberts", "Nature's Plus",
        "Now Foods", "Nutergia", "Pileje", "Nutersin", "Ana María Lajusticia", "IVB",
        "NHCO", "Vitanatur", "Eladiet", "ESI", "Drasanvi", "Santiveri",
        "El Granero Integral", "Soria Natural", "CFN", "100% Natural", "Phytogreen",
        "Salengei", "Equisalud", "Natures Bounty", "Healthaid", "Natysal",
    ],
    # MAGNESIO
    "magnesio": [
        "Ana María Lajusticia", "IVB Magnesio Total", "Magnesium", "NS Vitans Magnesio",
        "Aquilea Magnesio", "Magnesioboi", "Magnogene", "Solgar Magnesio", "Magmax",
        "Magnesia San Pellegrino",
    ],
    # COLÁGENO Y ARTICULACIONES
    "articulaciones": [
        "Colnatur", "Colagenova", "Epaplus", "Aquilea Articulaciones", "Arkoflex",
        "Movial", "Bonflex", "Condroitina", "Glucosamina", "Cartílago de tiburón",
        "MSM", "Cúrcuma", "Boswellia", "Harpagofito", "Vitanatur Collagen",
        "Drasanvi Colágeno", "Zentrum Colágeno", "Unique Pink Collagen", "Aminoflex",
        "Aminocollax", "IVB Articare", "Siik", "Marine Collagen", "Colageno Hidrolizado",
        "Peptan",
    ],
    # OMEGA 3
    "omega3": [
        "Nuadha", "Omacor", "Epaplus Omega 3", "Aquilea Omega 3", "Solgar Omega 3",
        "Nordic Naturals", "Möller's", "Nutergia", "Krill Oil", "Zeus Genesis",
        "IVB Omega", "Lamberts Omega", "Now Foods Omega", "Enerzona Omega", "Norsan",
    ],
    # HIERRO
    "hierro": [
        "Fero-Gradumet", "Tardyferon", "Floradix", "Ferroglicina", "Ferroprotect",
        "IVB Ferroplus", "Floravital", "Ferrosanol", "Iromax", "Fisiogen Ferro",
    ],
    # ENERGÍA Y CANSANCIO
    "energia_cansancio": [
        "Pharmaton", "Supradyn Energy", "Berocca Boost", "Leotron", "Astenolit",
        "Vitaminor", "Taurina", "Guaraná", "Ginseng", "Jalea Real", "Black Bee Pharmacy",
        "Royal Vit", "Apiserum", "Arkocápsulas Ginseng", "Vitanatur Energía",
        "Aminoenergy", "Maca", "Rhodiola", "Vitasprint", "Energynat",
    ],
    # MEMORIA Y CONCENTRACIÓN
    "memoria": [
        "Dememory", "Memorup", "Ginkgo Biloba", "Tavonin", "Tanakene", "Claudia",
        "Arkocápsulas Ginkgo", "NHCO Aminobrain", "IVB Memory-On", "Neuralex",
        "Intellán", "Mind Master", "Fosfatidilserina", "Bacopa", "Brain Plus",
    ],
    # SUEÑO E INSOMNIO
    "sueno_insomnio": [
        "Dormidina", "ZzzQuil", "Melamil", "NS Melatonina", "Aquilea Sueño", "Soñodor",
        "Sedatif PC", "Valeriana", "Passiflora", "Kneipp Valeriana",
        "Arkocápsulas Pasiflora", "Naturtierra Melatonina", "ESI Melatonina",
        "Drasanvi Melatonina", "Soria Natural Valeriana", "NS Soñaben", "Dormirel",
        "Lasea", "Serenity", "Rescue Night", "Epaplus Sleepcare", "Vitanatur Sueño",
        "Valdispert", "Nytol", "Circadin",
    ],
    # ESTRÉS Y ANSIEDAD
    "estres_ansiedad": [
        "Lasea", "Rescue Remedy", "NS Equirelax", "Aquilea Tranquilidad", "Sedatif PC",
        "Relaxing", "IVB Stress Off", "Rhodiola", "Ashwagandha", "L-Teanina", "GABA",
        "Triptófano", "5-HTP", "Valeriana", "Pasiflora",
    ],
    # EMBARAZO Y LACTANCIA
    "embarazo_prenatal": [
        "Natalben", "Femibion", "Gestagyn", "Seidivid", "Ovusitol", "Fertybiotic",
        "Inofolic", "Gestacare", "Maternum", "Natele", "Gynea", "Fertifem", "Lactanza",
        "Suavinex Natal", "Multicentrum Pre-Mama", "Solgar Prenatal", "Vitafolin",
        "Ácido fólico", "Yoduk", "Seidibion Mater",
    ],
    # MENOPAUSIA
    "menopausia": [
        "Flavia", "Ymea", "Estromineral", "Donna Plus", "Climafort", "Serelys",
        "Femikliman", "Cimicífuga", "Isoflavonas", "Aquilea Menopausia",
        "NS Menopausia", "IVB Menomaster", "Menopro", "Seidibion", "Aminopause",
        "Gynecomplex", "Ciminocta", "Menostop", "Menofem", "Onagra",
    ],
    # SALUD URINARIA / CISTITIS
    "infeccion_urinaria": [
        "Cistitus", "Cispren", "Urosens", "Aquilea Cistitis", "Arkocápsulas Arándano",
        "Monurelle", "Cysticlean", "Cystine", "Cystiberry", "Ellura", "Utipro",
        "Lactofem", "Feminabiane", "Cranberry", "D-Manosa",
    ],
    # SALUD OCULAR
    "ojo_seco": [
        "Thealoz", "Hyabak", "Hylo", "Systane", "Optiben", "Aquoral", "Ocular",
        "Lagrinet", "Lagroben", "Siccafluid", "Viscofresh", "Lipolac", "Lacryvisc",
        "Ojosbel", "Recugel", "Orcex", "Lephanet", "Visuectoine", "Ocualiv", "Estila",
    ],
    # SALUD OCULAR - SUPLEMENTOS
    "vision": [
        "Nutrof", "Oftan Macula", "Brudy Macula", "Vitreoclar", "Ocuvite",
        "Preservision", "Retinacaps", "IVB Vision", "Luteína", "Astaxantina",
    ],
    # HIGIENE DENTAL
    "higiene_dental": [
        "Lacer", "GUM", "Vitis", "Oral-B", "Sensodyne", "Parodontax", "Gingilacer",
        "Bexident", "Kin", "PHB", "Interprox", "TePe", "Curaprox", "Dentaid",
        "Fluocaril", "Elmex", "Lacerfresh", "Sensilacer", "Xerolacer", "Halita",
        "CB12", "Listerine", "Oraldine", "Oralkin", "Perio-Aid", "Clorhexidina",
        "Corsodyl", "Meridol", "Waterpik", "Philips Sonicare", "Kukident", "Corega",
        "Kin Oro", "Algasiv", "Aftamer",
    ],
    # AFTAS Y HERPES
    "aftas": [
        "Aftasone", "Aftaclear", "Aloclair", "Bexident Aftas", "Afta Repair",
    ],
    "herpes_labial": [
        "Zovirax", "Zovicrem", "Viruderm", "Zoviduo", "Compeed Herpes",
    ],
    # HIGIENE ÍNTIMA
    "higiene_intima": [
        "Cumlaude Lab", "Lactacyd", "Zelesse", "Isdin Woman", "Intimo", "Feminella",
        "Saugella", "Tantum Rosa", "Chilly", "Saforelle", "Enna", "Intimina",
        "Vagisil", "Durex Íntima", "Idracare", "Ainara", "Ginecanesten", "Ginexin",
        "Ozointim", "Palomacare", "Papilocare",
    ],
    # BEBÉS E INFANTIL - Alimentación
    "alimentacion_bebe": [
        "Hero Baby", "Nutribén", "Almirón", "Blemil", "Nidina", "NAN", "Enfamil",
        "Novalac", "Smileat", "Gerber", "Hipp",
    ],
    # BEBÉS E INFANTIL - Higiene y Cuidado
    "higiene_bebe": [
        "Mustela", "Suavinex", "Chicco", "Dodot", "Huggies", "Chelino", "Moltex",
        "Pingo", "Bio Baby", "ISDIN Baby Naturals", "Nutraisdin", "Mitosyl",
        "Bepanthol Baby", "Eryplast", "Halibut", "Lutsine", "Klorane Bebé",
        "Weleda Baby", "Cattier Baby", "Uriage Bebé",
    ],
    # BEBÉS E INFANTIL - Salud Infantil
    "salud_infantil": [
        "Apiretal", "Dalsy", "Reuteri Gotas", "Melamil", "Bivos", "NS Florabiotic Gotas",
        "Colimil", "Vicks BabyRub", "Rhinomer Baby", "Pedialac Suero",
    ],
    # LACTANCIA
    "lactancia": [
        "Suavinex", "Medela", "Philips Avent", "Chicco", "MAM", "Tommee Tippee",
        "NUK", "Lansinoh", "Lactanza", "Dr. Brown's",
    ],
    # ALERGIA
    "alergia": [
        "Ebastel", "Bactil", "Cetirizina", "Loratadina", "Bilaxten", "Xazal",
        "Aerius", "Telfast", "Abrilia", "Rinoebastel", "Reactine", "Clarityne",
        "Zyrtec", "Polaramine", "Atarax",
    ],
    # REPELENTES DE INSECTOS
    "repelente_insectos": [
        "Goibi", "Relec", "Autan", "Moskito Guard", "Cer8", "Isdin Antimosquitos",
        "Chicco Natural", "Parasidose", "Cinq sur Cinq", "Neositrín",
    ],
    # ANTIPIOJOS
    "piojos": [
        "OTC Antipiojos", "Neositrín", "Goibi Antipiojos", "Paranix", "Nyda",
        "Full Marks", "Itax", "Milice", "Permetrina", "Kitapic",
    ],
    # HIGIENE CORPORAL
    "hidratacion_corporal": [
        "Eucerin", "ISDIN", "Avène", "La Roche-Posay", "Bioderma", "Dove", "Nivea",
        "Neutrogena", "Johnson's", "Sanex", "Lactovit", "Mussvital",
        "Instituto Español", "Genové", "Germisdin", "Sebamed", "Inibsa", "Acofar",
        "Farline", "Atoderm", "Lipikar", "Xémose", "Topialyse", "Xeracalm", "Ureadin",
    ],
    # PIEL ATÓPICA
    "piel_atopica": [
        "Lipikar", "Atoderm", "Xeracalm", "Xémose", "Eucerin AtopiControl",
        "Topialyse", "Leti AT4", "Nutratopic", "Exomega", "Stelatopia", "Primalba",
        "Bepanthol Sensidaily", "Be+ Atopia", "Rilastil Xerolact", "Sebamed",
    ],
    # ACNÉ
    "acne": [
        "Effaclar", "Sebium", "Cleanance", "Acniben", "Martiderm Acniover",
        "Sesderma Azelac", "Peroxiben", "Benzac", "Panoxyl", "Acmed", "Azeane",
        "Boreade", "Keracnyl", "Normaderm", "Eucerin DermoPure",
    ],
    # HEMORROIDES
    "hemorroides": [
        "Hemoal", "Hemorrane", "Ruscus Llorens", "Proctoial", "Proctolog", "Faktu",
        "Detramax", "Anso", "Sante Verte Hemoless", "Proctosan",
    ],
    # CIRCULACIÓN / PIERNAS CANSADAS
    "circulacion_piernas": [
        "Daflon", "Variliv", "Venoruton", "Venovid", "Flebinec", "Venosmil",
        "Flebostasin", "Martiderm Legvass", "Aquilea Piernas Ligeras", "Veinotonyl",
        "Antistax", "Ginkor", "Venoplus", "Venen", "Edemox",
    ],
    # HERIDAS Y CICATRIZACIÓN
    "heridas_apositos": [
        "Betadine", "Cristalmina", "Clorhexidina", "Blastoestimulina", "Cicaplast",
        "Cicalfate", "Cicabio", "Nutracel", "Repavar", "Trofolastin", "Mederma",
        "Dermatix", "Kelo-Cote", "Contractubex", "Bio-Oil", "Suaviskin", "Linitul",
        "Tulgrasum", "Iruxol", "Lacer Gel Clorhexidina",
    ],
    # HONGOS / MICOSIS
    "hongos_pies": [
        "Canesten", "Lamisil", "Fungarest", "Mycoben", "Farmafeet", "PO Polvos Pédicos",
        "Funsol", "Zeliderm", "Terbinafina", "Ketoconazol",
    ],
    # VERRUGAS Y CALLOS
    "verrugas_callos": [
        "ISDIN Antiverrugas", "Wartner", "Scholl", "Compeed Callos", "Salvelox Callos",
        "Urgo Verrugas", "Excilor", "Duofilm", "Pointts", "Farmapodo",
    ],
    # INCONTINENCIA
    "incontinencia": [
        "Tena", "Lindor", "Ausonia Discreet", "iD", "Molicare", "Depend", "Attends",
        "Abena", "Ontex", "Hartmann",
    ],
    # CUIDADO LABIOS
    "cuidado_labios": [
        "Carmex", "Blistex", "Labello", "Neutrogena Labios", "ISDIN Labial",
        "Avène Cold Cream Labios", "La Roche-Posay Cicaplast Lèvres", "Repavar Labial",
        "Letibalm", "Eucerin Lip Active", "Vichy Aqualia Labios", "Burts Bees", "EOS",
        "Himalaya", "Be+ Med Labios",
    ],
    # MAREO / VIAJE
    "mareo": [
        "Biodramina", "Normostop", "Sweetsin Stop Mareo", "Cinfamar", "Navicalm",
    ],
    # TABAQUISMO
    "tabaquismo": [
        "Nicotinell", "Niquitin", "Nicorette", "Todacitan", "Champix",
    ],
    # VETERINARIA
    "veterinaria": [
        "Scalibor", "Frontline", "Advantix", "Nexgard", "Bravecto", "Seresto",
        "Fortekor", "Otomax", "Recicort", "Upcard",
    ],
    # ORTOPEDIA / BOTIQUÍN
    "material_sanitario": [
        "Farmalastic", "Orliman", "Futuro", "Actimove", "Emo", "Varisan", "Medilast",
        "Aposan", "Hansaplast", "Compeed", "Tiritas", "Leukoplast", "Cosmopor",
        "Nexcare", "Omron",
    ],
    # HOMEOPATÍA
    "homeopatia": [
        "Boiron", "Lehning", "Weleda", "DHU", "Heel", "Schwabe", "Nelsons", "A. Vogel",
        "Traumeel", "Oscillococcinum",
    ],
}

# Marcas ya existentes en el servicio (de BRAND_PRIORITY)
EXISTING_BRANDS = {
    'la roche posay', 'la roche-posay', 'ivb wellness', 'gema herrerias',
    'arturo alba', 'dr arthouros alba', 'dr arthouros',
    'gine-canesten', 'gine canesten', 'oral-b', 'oral b', 'a-derma',
    'huile prodigieuse', 'heliocare', 'isdin', 'bioderma', 'eucerin', 'avene',
    'sesderma', 'cerave', 'neutrogena', 'medik8', 'esthederm', 'cumlaude',
    'klorane', 'be+', 'nuxe', 'caudalie', 'filorga', 'uriage', 'aderma',
    'sensodyne', 'parodontax', 'gum', 'lacer', 'kukident', 'corega', 'kin',
    'bexident', 'phb', 'vitis', 'fluocaril', 'olistic', 'arkopharma',
    'arkocapsulas', 'arkovital', 'aquilea', 'solgar', 'centrum', 'multicentrum',
    'supradyn', 'ivb', 'ns', 'nhco', 'gh', 'nutracel', 'thealoz', 'hylo',
    'blemil', 'pedialac', 'mitosyl', 'bivos', 'nutriben', 'nutribén', 'mustela',
    'dodot', 'suavinex', 'voltadol', 'frenadol', 'fluimucil', 'cristalmina',
    'almax', 'zzzquil', 'ibudol', 'respibien', 'vicks', 'vaporub', 'omron',
    'predictor', 'dulcosoft', 'megalevure', 'daflon', 'fortasec', 'nurofen',
    'thrombocid', 'dormidina', 'bucomax', 'dulcolaxo', 'fisiocrem', 'compeed',
    'filvit', 'lactoflora', 'afterbite', 'after bite', 'hansaplast',
    'normogrip', 'normotus', 'frionex', 'otocerum', 'medebiotin', 'reflex',
    'diclokern', 'espididol', 'hidroxil', 'ellaone', 'flutox', 'variliv',
    'salvacolina', 'zarbees', 'tavonin', 'venovid', 'nuadha', 'natalben',
    'micralax', 'aero red', 'aerored', 'fisiogen', 'fresubin', 'fontactiv',
    'cinfamucol', 'cinfadol', 'cinfatos', 'melilax', 'seidivid', 'flogoprofen',
    'difenadol', 'flavia', 'actron', 'ibufen', 'omereflux', 'exelvit', 'relive',
    'zoviduo', 'optiben', 'lindor', 'kabi', 'aminocomplex', 'aminohair',
    'vichy',  # Añadir alias comunes
}


def normalize_brand(brand: str) -> str:
    """Normaliza un nombre de marca para comparación."""
    # Limpiar caracteres especiales
    brand = brand.lower().strip()
    # Eliminar acentos comunes
    replacements = {
        'á': 'a', 'é': 'e', 'í': 'i', 'ó': 'o', 'ú': 'u',
        'ñ': 'n', 'ü': 'u',
    }
    for old, new in replacements.items():
        brand = brand.replace(old, new)
    return brand


def extract_all_brands():
    """Extrae todas las marcas únicas de la lista del usuario."""
    all_brands = set()
    brand_to_category = {}

    for category, brands in USER_BRANDS.items():
        for brand in brands:
            normalized = normalize_brand(brand)
            # Limpiar marcas compuestas como "ISDIN Fotoprotector" -> "isdin"
            # Solo tomamos la primera palabra si es una marca conocida
            if ' ' in normalized:
                first_word = normalized.split()[0]
                # Si la primera palabra es la marca principal, usar eso
                if first_word in EXISTING_BRANDS:
                    continue  # Ya existe
            all_brands.add(normalized)
            if normalized not in brand_to_category:
                brand_to_category[normalized] = category

    return all_brands, brand_to_category


def find_new_brands():
    """Encuentra marcas nuevas que no existen en el servicio."""
    all_brands, brand_to_category = extract_all_brands()

    new_brands = []
    for brand in all_brands:
        # Verificar si la marca o alguna variante existe
        is_existing = False
        for existing in EXISTING_BRANDS:
            if brand == existing or brand in existing or existing in brand:
                is_existing = True
                break
        if not is_existing:
            new_brands.append((brand, brand_to_category.get(brand, 'unknown')))

    return sorted(new_brands, key=lambda x: x[0])


def generate_brand_code(new_brands):
    """Genera el código Python para añadir las nuevas marcas."""
    print("=" * 60)
    print("NUEVAS MARCAS A AÑADIR")
    print("=" * 60)
    print(f"\nTotal marcas en lista usuario: {sum(len(b) for b in USER_BRANDS.values())}")
    print(f"Marcas únicas (normalizadas): {len(extract_all_brands()[0])}")
    print(f"Marcas existentes en servicio: {len(EXISTING_BRANDS)}")
    print(f"Nuevas marcas a añadir: {len(new_brands)}")

    print("\n" + "=" * 60)
    print("LISTA DE NUEVAS MARCAS POR CATEGORÍA")
    print("=" * 60)

    # Agrupar por categoría
    by_category = {}
    for brand, category in new_brands:
        if category not in by_category:
            by_category[category] = []
        by_category[category].append(brand)

    for category, brands in sorted(by_category.items()):
        print(f"\n{category.upper()} ({len(brands)} marcas):")
        for brand in sorted(brands):
            print(f"  - {brand}")

    return new_brands


if __name__ == "__main__":
    new_brands = find_new_brands()
    generate_brand_code(new_brands)
