﻿#!/usr/bin/env python
"""
Script para migrar datos desde HomogeneousGroup a la nueva arquitectura
HomogeneousGroupMaster + PharmacyHomogeneousMetrics
"""

import argparse
import logging
import os
import sys
from typing import Dict, List

# Añadir el directorio backend al path
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), "..")))

from sqlalchemy.orm import Session

from app.database import SessionLocal
from app.models.homogeneous_group import HomogeneousGroup
from app.models.homogeneous_group_master import HomogeneousGroupMaster
from app.models.pharmacy_homogeneous_metrics import PharmacyHomogeneousMetrics
from app.utils.datetime_utils import utc_now

# Configurar logging
logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


def extract_master_data(groups: List[HomogeneousGroup]) -> Dict:
    """
    Extrae datos maestros comunes de un conjunto de grupos homogéneos
    """
    # Usar el primer grupo como base y validar consistencia
    first = groups[0]

    # Recopilar todos los laboratorios únicos
    all_labs = set()
    for group in groups:
        if group.partner_laboratories:
            all_labs.update(group.partner_laboratories)

    # Tomar el PVL más reciente
    latest_pvl = None
    latest_date = None
    for group in groups:
        if group.calculated_pvl:
            if not latest_date or (group.updated_at and group.updated_at > latest_date):
                latest_pvl = group.calculated_pvl
                latest_date = group.updated_at

    return {
        "homogeneous_code": first.homogeneous_code,
        "homogeneous_name": first.homogeneous_name,
        "active_ingredient": first.main_active_ingredient,
        "reference_price": latest_pvl,
        "total_products": (
            max(g.total_labs_in_group for g in groups if g.total_labs_in_group)
            if any(g.total_labs_in_group for g in groups)
            else 0
        ),
        "all_laboratories": sorted(list(all_labs)),
        "therapeutic_group": first.therapeutic_group,
        "requires_prescription": first.requires_prescription,
        "last_nomenclator_update": utc_now(),
    }


def migrate_data(db: Session, dry_run: bool = True):
    """
    Migra datos desde HomogeneousGroup a la nueva arquitectura
    """
    logger.info("Iniciando migración de datos de HomogeneousGroup...")

    # Obtener todos los grupos homogéneos únicos
    all_groups = db.query(HomogeneousGroup).all()
    logger.info(f"Encontrados {len(all_groups)} registros de HomogeneousGroup")

    # Agrupar por código homogéneo
    groups_by_code = {}
    for group in all_groups:
        if group.homogeneous_code not in groups_by_code:
            groups_by_code[group.homogeneous_code] = []
        groups_by_code[group.homogeneous_code].append(group)

    logger.info(f"Códigos homogéneos únicos: {len(groups_by_code)}")

    if dry_run:
        logger.info("=== MODO DRY RUN - No se realizarán cambios ===")

    masters_created = 0
    metrics_created = 0
    errors = []

    for code, groups in groups_by_code.items():
        try:
            # 1. Crear o actualizar el registro maestro
            master_data = extract_master_data(groups)

            if not dry_run:
                # Verificar si ya existe
                existing_master = db.query(HomogeneousGroupMaster).filter_by(homogeneous_code=code).first()

                if existing_master:
                    # Actualizar
                    for key, value in master_data.items():
                        setattr(existing_master, key, value)
                    logger.debug(f"Actualizado master: {code}")
                else:
                    # Crear nuevo
                    new_master = HomogeneousGroupMaster(**master_data)
                    db.add(new_master)
                    masters_created += 1
                    logger.debug(f"Creado master: {code}")
            else:
                masters_created += 1
                logger.debug(f"[DRY RUN] Crearía master: {code}")

            # 2. Crear métricas por farmacia
            for group in groups:
                if group.pharmacy_id:  # Solo si tiene farmacia asociada
                    metrics_data = {
                        "pharmacy_id": group.pharmacy_id,
                        "homogeneous_code": group.homogeneous_code,
                        "partner_laboratories": group.partner_laboratories or [],
                        "total_units_sold": group.total_units_sold or 0,
                        "partner_units_sold": group.partner_units_sold or 0,
                        "total_revenue": group.total_revenue or 0,
                        "partner_revenue": group.partner_revenue or 0,
                        "last_sale_date": group.last_sales_date,
                        "data_period_start": group.data_period_start,
                        "data_period_end": group.data_period_end,
                    }

                    if not dry_run:
                        # Verificar si ya existe
                        existing_metrics = (
                            db.query(PharmacyHomogeneousMetrics)
                            .filter_by(pharmacy_id=group.pharmacy_id, homogeneous_code=group.homogeneous_code)
                            .first()
                        )

                        if existing_metrics:
                            # Actualizar
                            for key, value in metrics_data.items():
                                setattr(existing_metrics, key, value)
                            logger.debug(f"Actualizado metrics: {code} para farmacia {group.pharmacy_id}")
                        else:
                            # Crear nuevo
                            new_metrics = PharmacyHomogeneousMetrics(**metrics_data)
                            db.add(new_metrics)
                            metrics_created += 1
                            logger.debug(f"Creado metrics: {code} para farmacia {group.pharmacy_id}")
                    else:
                        metrics_created += 1
                        logger.debug(f"[DRY RUN] Crearía metrics: {code} para farmacia {group.pharmacy_id}")

        except Exception as e:
            error_msg = f"Error procesando código {code}: {str(e)}"
            logger.error(error_msg)
            errors.append(error_msg)

    if not dry_run:
        try:
            db.commit()
            logger.info("Cambios confirmados en la base de datos")
        except Exception as e:
            db.rollback()
            logger.error(f"Error al confirmar cambios: {str(e)}")
            raise

    # Resumen
    logger.info("=== RESUMEN DE MIGRACIÓN ===")
    logger.info(f"Registros maestros creados/actualizados: {masters_created}")
    logger.info(f"Registros de métricas creados/actualizados: {metrics_created}")
    if errors:
        logger.warning(f"Errores encontrados: {len(errors)}")
        for error in errors[:5]:  # Mostrar primeros 5 errores
            logger.warning(f"  - {error}")

    return {"masters_created": masters_created, "metrics_created": metrics_created, "errors": errors}


def verify_migration(db: Session):
    """
    Verifica que la migración se completó correctamente
    """
    logger.info("Verificando migración...")

    # Contar registros
    old_count = db.query(HomogeneousGroup).count()
    master_count = db.query(HomogeneousGroupMaster).count()
    metrics_count = db.query(PharmacyHomogeneousMetrics).count()

    logger.info(f"Registros originales (HomogeneousGroup): {old_count}")
    logger.info(f"Registros maestros (HomogeneousGroupMaster): {master_count}")
    logger.info(f"Registros métricas (PharmacyHomogeneousMetrics): {metrics_count}")

    # Verificar integridad referencial
    orphan_metrics = (
        db.query(PharmacyHomogeneousMetrics)
        .filter(~PharmacyHomogeneousMetrics.homogeneous_code.in_(db.query(HomogeneousGroupMaster.homogeneous_code)))
        .count()
    )

    if orphan_metrics > 0:
        logger.warning(f"⚠️ Encontradas {orphan_metrics} métricas sin maestro correspondiente")
    else:
        logger.info("✅ Todas las métricas tienen su maestro correspondiente")

    return {
        "old_count": old_count,
        "master_count": master_count,
        "metrics_count": metrics_count,
        "orphan_metrics": orphan_metrics,
    }


def main():
    parser = argparse.ArgumentParser(description="Migrar datos de HomogeneousGroup a nueva arquitectura")
    parser.add_argument("--dry-run", action="store_true", help="Simular migración sin cambios")
    parser.add_argument("--verify", action="store_true", help="Solo verificar estado de migración")
    args = parser.parse_args()

    db = SessionLocal()

    try:
        if args.verify:
            verify_migration(db)
        else:
            migrate_data(db, dry_run=args.dry_run)
            if not args.dry_run:
                verify_migration(db)
    finally:
        db.close()


if __name__ == "__main__":
    main()
