#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Script para cargar listados oficiales de prescripcion desde Excel a BD.

Carga desde pf-fichero-1-noviembrev2xlsx.xlsx:
- Dietas (1,085 productos) -> DIETOTERAPICOS
- Tiras reactivas (27 productos) -> TIRAS_REACTIVAS_GLUCOSA
- Efectos y accesorios (4,679 productos):
  - ABSORB INC* -> INCONTINENCIA_FINANCIADA (456)
  - MEDIA*|MUNEQUERA*|MUSLERA*|RODILLERA*|TOBILLERA* -> ORTOPEDIA_FINANCIADA (1,409)
  - Resto -> EFECTOS_FINANCIADOS (2,814)

Uso:
    python scripts/load_prescription_references.py --file /app/pf-fichero-1-noviembrev2xlsx.xlsx
    python scripts/load_prescription_references.py --file /app/pf-fichero-1-noviembrev2xlsx.xlsx --truncate
"""
import argparse
import io
import sys
from pathlib import Path

# Fix stdout encoding for Windows console
if sys.platform == "win32":
    sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding='utf-8', errors='replace')

import pandas as pd

# Añadir raíz del proyecto al PYTHONPATH
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.database import SessionLocal
from app.models.enums import PrescriptionCategory
from app.models.prescription_reference_list import PrescriptionReferenceList
from app.models.product_catalog import ProductCatalog


def ensure_product_catalog(session, national_code: str, product_name: str, category: PrescriptionCategory, reference_source: str):
    """
    Crear entrada en ProductCatalog si no existe.

    Issue #449: Los productos de prescription_reference_list necesitan existir en
    ProductCatalog para aparecer en analytics (joins con SalesEnrichment).
    """
    existing = session.query(ProductCatalog).filter(
        ProductCatalog.national_code == national_code
    ).first()

    if not existing:
        product = ProductCatalog(
            national_code=national_code,
            cima_nombre_comercial=product_name,
            nomen_nombre=product_name,
            xfarma_prescription_category=category,
            data_sources=f"prescription_reference_list:{reference_source}",
        )
        session.add(product)


def load_dietas(df: pd.DataFrame, session) -> int:
    """Cargar productos dietoterapéuticos."""
    count = 0
    for _, row in df.iterrows():
        national_code = str(row['CN'])
        product_name = row['DESCRIPCIÓN '][:500]
        category = PrescriptionCategory.DIETOTERAPICOS

        ref = PrescriptionReferenceList(
            national_code=national_code,
            product_name=product_name,
            product_description_long=row.get('descripcion2', '')[:500] if pd.notna(row.get('descripcion2')) else None,
            category=category.value,
            reference_source='Dietas',
            pf_code=str(row['PF']) if pd.notna(row['PF']) else None,
            pvp_iva=row.get('PMF') if pd.notna(row.get('PMF')) else None,
            status=row.get('situacion ') if pd.notna(row.get('situacion ')) else None
        )
        session.add(ref)

        # Issue #449: También crear ProductCatalog
        ensure_product_catalog(session, national_code, product_name, category, 'Dietas')

        count += 1
    return count


def load_tiras_reactivas(df: pd.DataFrame, session) -> int:
    """Cargar tiras reactivas de glucosa."""
    count = 0
    for _, row in df.iterrows():
        national_code = str(row['CN'])
        product_name = row['DESCRIPCIÓN '][:500]
        category = PrescriptionCategory.TIRAS_REACTIVAS_GLUCOSA

        ref = PrescriptionReferenceList(
            national_code=national_code,
            product_name=product_name,
            category=category.value,
            reference_source='Tiras reactivas',
            pf_code=str(row['PF']) if pd.notna(row['PF']) else None,
            pvp_iva=row.get('PMF') if pd.notna(row.get('PMF')) else None
        )
        session.add(ref)

        # Issue #449: También crear ProductCatalog
        ensure_product_catalog(session, national_code, product_name, category, 'Tiras reactivas')

        count += 1
    return count


def load_efectos_accesorios(df: pd.DataFrame, session) -> dict:
    """Cargar efectos y accesorios (incontinencia, ortopedia, otros)."""
    counts = {'incontinencia': 0, 'ortopedia': 0, 'efectos': 0}

    for _, row in df.iterrows():
        national_code = str(row['CN'])
        product_name = row['DESCRIPCIÓN '][:500]
        col5 = str(row.get('Unnamed: 4', ''))

        # Clasificar por columna 5
        if col5.startswith('ABSORB INC'):
            category_enum = PrescriptionCategory.INCONTINENCIA_FINANCIADA
            counts['incontinencia'] += 1
        elif any(col5.startswith(prefix) for prefix in ['MEDIA', 'MUÑEQUERA', 'MUSLERA', 'RODILLERA', 'TOBILLERA']):
            category_enum = PrescriptionCategory.ORTOPEDIA_FINANCIADA
            counts['ortopedia'] += 1
        else:
            category_enum = PrescriptionCategory.EFECTOS_FINANCIADOS
            counts['efectos'] += 1

        ref = PrescriptionReferenceList(
            national_code=national_code,
            product_name=product_name,
            product_description_long=col5[:500] if col5 else None,
            category=category_enum.value,
            reference_source='Efectos y accesorios',
            pf_code=str(row['PF']) if pd.notna(row['PF']) else None,
            pvp_iva=row.get('PVP + iva') if pd.notna(row.get('PVP + iva')) else None,
            status=row.get('situación') if pd.notna(row.get('situación')) else None
        )
        session.add(ref)

        # Issue #449: También crear ProductCatalog
        ensure_product_catalog(session, national_code, product_name, category_enum, 'Efectos y accesorios')

    return counts


def main():
    parser = argparse.ArgumentParser(description="Cargar listados oficiales de prescripción")
    parser.add_argument('--file', required=True, help="Path al archivo Excel")
    parser.add_argument('--truncate', action='store_true', help="Truncar tabla antes de cargar")
    args = parser.parse_args()

    print("="*80)
    print("CARGA DE LISTADOS DE REFERENCIA DE PRESCRIPCIÓN")
    print("="*80)
    print(f"Archivo: {args.file}\n")

    # Iniciar sesión
    session = SessionLocal()

    try:
        # Truncar si se solicita
        if args.truncate:
            count = session.query(PrescriptionReferenceList).delete()
            session.commit()
            print(f"[TRUNCATE] Eliminados {count:,} registros existentes\n")

        # Leer Excel
        xl = pd.ExcelFile(args.file)

        # 1. Cargar DIETOTERAPICOS
        print("[1/3] Cargando DIETOTERAPICOS...")
        df_dietas = pd.read_excel(args.file, sheet_name='Dietas')
        dietas_count = load_dietas(df_dietas, session)
        print(f"      -> {dietas_count:,} productos")

        # 2. Cargar TIRAS_REACTIVAS_GLUCOSA
        print("\n[2/3] Cargando TIRAS_REACTIVAS_GLUCOSA...")
        df_tiras = pd.read_excel(args.file, sheet_name='Tiras reactivas')
        tiras_count = load_tiras_reactivas(df_tiras, session)
        print(f"      -> {tiras_count:,} productos")

        # 3. Cargar EFECTOS Y ACCESORIOS
        print("\n[3/3] Cargando EFECTOS Y ACCESORIOS...")
        df_efectos = pd.read_excel(args.file, sheet_name='Efectos y accesorios')
        efectos_counts = load_efectos_accesorios(df_efectos, session)
        print(f"      -> INCONTINENCIA_FINANCIADA: {efectos_counts['incontinencia']:,}")
        print(f"      -> ORTOPEDIA_FINANCIADA: {efectos_counts['ortopedia']:,}")
        print(f"      -> EFECTOS_FINANCIADOS: {efectos_counts['efectos']:,}")

        # 4. Cargar FORMULAS_MAGISTRALES y VACUNAS (códigos especiales)
        print("\n[4/4] Agregando códigos especiales...")
        # Fórmulas magistrales (CN 500017)
        session.add(PrescriptionReferenceList(
            national_code='500017',
            product_name='FORMULAS MAGISTRALES',
            category=PrescriptionCategory.FORMULAS_MAGISTRALES.value,
            reference_source='Códigos especiales'
        ))
        # Vacunas hipoalergénicas (CN 500009)
        session.add(PrescriptionReferenceList(
            national_code='500009',
            product_name='EXTRACTOS HIPOSENSIBILIZANTES Y VACUNAS BACTERIANAS',
            category=PrescriptionCategory.VACUNAS_INDIVIDUALIZADAS.value,
            reference_source='Códigos especiales'
        ))
        print(f"      -> 2 codigos especiales")

        # Commit
        session.commit()

        # Resumen
        total = dietas_count + tiras_count + sum(efectos_counts.values()) + 2
        print(f"\n{'='*80}")
        print(f"CARGA COMPLETADA EXITOSAMENTE")
        print(f"{'='*80}")
        print(f"Total productos cargados: {total:,}")
        print(f"\nDesglose:")
        print(f"  - DIETOTERAPICOS: {dietas_count:,}")
        print(f"  - TIRAS_REACTIVAS_GLUCOSA: {tiras_count:,}")
        print(f"  - INCONTINENCIA_FINANCIADA: {efectos_counts['incontinencia']:,}")
        print(f"  - ORTOPEDIA_FINANCIADA: {efectos_counts['ortopedia']:,}")
        print(f"  - EFECTOS_FINANCIADOS: {efectos_counts['efectos']:,}")
        print(f"  - Códigos especiales: 2")
        print(f"{'='*80}")

    except Exception as e:
        session.rollback()
        print(f"\n[ERROR] {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)
    finally:
        session.close()


if __name__ == "__main__":
    main()
