"""
Script de investigación: Gap de sincronización CIMA

Detecta discrepancias entre la API de CIMA y el catálogo local.

CONTEXTO CRÍTICO:
- API CIMA reporta ~67,493 presentaciones
- Catálogo local tiene 67,417 productos con datos CIMA
- Checkpoint muestra 77,400 productos procesados hasta página 313
- VOLTADOL FORTE (CNs 763900, 768260) SÍ existe en catálogo local con data_sources='cima'

HIPÓTESIS A INVESTIGAR:
1. ¿Hay productos duplicados que se procesan múltiples veces?
2. ¿Hay productos que fallan en el mapeo silenciosamente?
3. ¿El checkpoint cuenta productos que se saltan (skip)?
4. ¿Hay errores en el proceso de upsert bulk que no se registran?

Ejecutar:
    docker-compose exec backend python backend/scripts/investigate_cima_gap.py
"""

import os
import sys
from collections import defaultdict, Counter
from decimal import Decimal

# Añadir path del proyecto
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from sqlalchemy import func, and_, or_, text
from sqlalchemy.orm import Session

from backend.app.database import SessionLocal
from backend.app.models.product_catalog import ProductCatalog
from backend.app.models.system_status import SystemStatus


def check_cima_status():
    """Verifica el estado del sistema CIMA."""
    db = SessionLocal()
    try:
        print("\n" + "=" * 80)
        print("1. ESTADO DE SINCRONIZACIÓN CIMA")
        print("=" * 80)

        # Buscar componente CIMA
        from backend.app.models.enums import SystemComponent
        cima_status = db.query(SystemStatus).filter(
            SystemStatus.component == SystemComponent.CIMA
        ).first()

        if cima_status:
            print(f"\n✅ Componente CIMA encontrado:")
            print(f"   Status: {cima_status.status}")
            print(f"   Mensaje: {cima_status.message}")
            print(f"   Checkpoint página: {cima_status.checkpoint_page}")
            print(f"   Checkpoint timestamp: {cima_status.checkpoint_timestamp}")
            print(f"   Total items: {cima_status.total_items}")
            print(f"   Items procesados: {cima_status.processed_items}")
            print(f"   Errores: {cima_status.error_count}")

            if cima_status.checkpoint_data:
                import json
                try:
                    checkpoint_data = json.loads(cima_status.checkpoint_data)
                    print(f"\n   Checkpoint data:")
                    for key, value in checkpoint_data.items():
                        print(f"     {key}: {value}")
                except:
                    print(f"   Checkpoint data (raw): {cima_status.checkpoint_data[:200]}")
        else:
            print("\n❌ No se encontró registro de CIMA en system_status")

    finally:
        db.close()


def analyze_catalog_cima_data():
    """Analiza los datos CIMA en el catálogo local."""
    db = SessionLocal()
    try:
        print("\n" + "=" * 80)
        print("2. ANÁLISIS DE DATOS CIMA EN CATÁLOGO LOCAL")
        print("=" * 80)

        # Total de productos
        total_products = db.query(func.count(ProductCatalog.id)).scalar()
        print(f"\nTotal productos en catálogo: {total_products:,}")

        # Productos con datos CIMA
        cima_products = db.query(func.count(ProductCatalog.id)).filter(
            ProductCatalog.data_sources.like('%cima%')
        ).scalar()
        print(f"Productos con datos CIMA: {cima_products:,}")

        # Productos SOLO CIMA
        cima_only = db.query(func.count(ProductCatalog.id)).filter(
            ProductCatalog.data_sources == 'cima'
        ).scalar()
        print(f"Productos SOLO CIMA: {cima_only:,}")

        # Productos CIMA + Nomenclator
        cima_nomen = db.query(func.count(ProductCatalog.id)).filter(
            or_(
                ProductCatalog.data_sources == 'cima,nomenclator',
                ProductCatalog.data_sources == 'nomenclator,cima'
            )
        ).scalar()
        print(f"Productos CIMA + Nomenclator: {cima_nomen:,}")

        # Verificar campos CIMA poblados
        print("\n📊 Campos CIMA poblados:")
        cima_fields = [
            'cima_nombre_comercial',
            'cima_nregistro',
            'cima_laboratorio_titular',
            'cima_requiere_receta',
            'cima_estado_registro',
            'cima_atc_code',
            'cima_principios_activos'
        ]

        for field in cima_fields:
            count = db.query(func.count(ProductCatalog.id)).filter(
                and_(
                    ProductCatalog.data_sources.like('%cima%'),
                    getattr(ProductCatalog, field).isnot(None)
                )
            ).scalar()
            pct = (count / cima_products * 100) if cima_products > 0 else 0
            print(f"   {field}: {count:,} ({pct:.1f}%)")

    finally:
        db.close()


def check_for_duplicates():
    """Verifica si hay códigos nacionales duplicados."""
    db = SessionLocal()
    try:
        print("\n" + "=" * 80)
        print("3. VERIFICACIÓN DE DUPLICADOS")
        print("=" * 80)

        # Buscar CNs duplicados
        duplicates = db.query(
            ProductCatalog.national_code,
            func.count(ProductCatalog.id).label('count')
        ).filter(
            and_(
                ProductCatalog.national_code.isnot(None),
                ProductCatalog.national_code != ''
            )
        ).group_by(
            ProductCatalog.national_code
        ).having(
            func.count(ProductCatalog.id) > 1
        ).all()

        if duplicates:
            print(f"\n❌ ENCONTRADOS {len(duplicates)} CNs DUPLICADOS:")
            for cn, count in duplicates[:20]:  # Mostrar primeros 20
                print(f"   CN {cn}: {count} registros")

                # Mostrar detalles de los duplicados
                dups = db.query(ProductCatalog).filter(
                    ProductCatalog.national_code == cn
                ).all()
                for dup in dups:
                    print(f"     - ID: {dup.id}, Sources: {dup.data_sources}, Nombre: {dup.cima_nombre_comercial or dup.nomen_nombre}")
        else:
            print("\n✅ No se encontraron CNs duplicados")

    finally:
        db.close()


def check_mapping_issues():
    """Verifica productos con campos vacíos que deberían estar poblados."""
    db = SessionLocal()
    try:
        print("\n" + "=" * 80)
        print("4. VERIFICACIÓN DE PROBLEMAS DE MAPEO")
        print("=" * 80)

        # Productos CIMA sin nombre comercial (crítico)
        missing_name = db.query(func.count(ProductCatalog.id)).filter(
            and_(
                ProductCatalog.data_sources.like('%cima%'),
                or_(
                    ProductCatalog.cima_nombre_comercial.is_(None),
                    ProductCatalog.cima_nombre_comercial == ''
                )
            )
        ).scalar()

        if missing_name > 0:
            print(f"\n⚠️ Productos CIMA sin nombre comercial: {missing_name:,}")

            # Mostrar ejemplos
            examples = db.query(ProductCatalog).filter(
                and_(
                    ProductCatalog.data_sources.like('%cima%'),
                    or_(
                        ProductCatalog.cima_nombre_comercial.is_(None),
                        ProductCatalog.cima_nombre_comercial == ''
                    )
                )
            ).limit(10).all()

            print("\n   Ejemplos:")
            for p in examples:
                print(f"   CN: {p.national_code}, Nregistro: {p.cima_nregistro}, Sources: {p.data_sources}")
        else:
            print("\n✅ Todos los productos CIMA tienen nombre comercial")

        # Productos CIMA sin nregistro
        missing_nregistro = db.query(func.count(ProductCatalog.id)).filter(
            and_(
                ProductCatalog.data_sources.like('%cima%'),
                or_(
                    ProductCatalog.cima_nregistro.is_(None),
                    ProductCatalog.cima_nregistro == ''
                )
            )
        ).scalar()

        print(f"\n   Productos CIMA sin nregistro: {missing_nregistro:,}")

    finally:
        db.close()


def verify_voltadol():
    """Verifica específicamente los productos VOLTADOL."""
    db = SessionLocal()
    try:
        print("\n" + "=" * 80)
        print("5. VERIFICACIÓN ESPECÍFICA: VOLTADOL FORTE")
        print("=" * 80)

        voltadol_cns = ['763900', '768260']

        for cn in voltadol_cns:
            product = db.query(ProductCatalog).filter(
                ProductCatalog.national_code == cn
            ).first()

            if product:
                print(f"\n✅ CN {cn} ENCONTRADO:")
                print(f"   ID: {product.id}")
                print(f"   Nombre comercial: {product.cima_nombre_comercial}")
                print(f"   Laboratorio: {product.cima_laboratorio_titular}")
                print(f"   Nregistro: {product.cima_nregistro}")
                print(f"   Data sources: {product.data_sources}")
                print(f"   Requiere receta: {product.cima_requiere_receta}")
                print(f"   Estado: {product.cima_estado_registro}")
                print(f"   Updated at: {product.updated_at}")
            else:
                print(f"\n❌ CN {cn} NO ENCONTRADO")

    finally:
        db.close()


def analyze_data_sources_distribution():
    """Analiza la distribución de data_sources."""
    db = SessionLocal()
    try:
        print("\n" + "=" * 80)
        print("6. DISTRIBUCIÓN DE DATA_SOURCES")
        print("=" * 80)

        distribution = db.query(
            ProductCatalog.data_sources,
            func.count(ProductCatalog.id).label('count')
        ).group_by(
            ProductCatalog.data_sources
        ).order_by(
            func.count(ProductCatalog.id).desc()
        ).all()

        print(f"\nDistribución de fuentes de datos:")
        print("-" * 60)

        total = sum(count for _, count in distribution)

        for source, count in distribution:
            pct = (count / total * 100) if total > 0 else 0
            print(f"   {source or 'NULL':<40}: {count:>8,} ({pct:>5.1f}%)")

    finally:
        db.close()


def check_cima_estado_registro():
    """Analiza el campo cima_estado_registro."""
    db = SessionLocal()
    try:
        print("\n" + "=" * 80)
        print("7. ANÁLISIS DE cima_estado_registro")
        print("=" * 80)

        # Productos CIMA por estado
        estados = db.query(
            ProductCatalog.cima_estado_registro,
            func.count(ProductCatalog.id).label('count')
        ).filter(
            ProductCatalog.data_sources.like('%cima%')
        ).group_by(
            ProductCatalog.cima_estado_registro
        ).order_by(
            func.count(ProductCatalog.id).desc()
        ).all()

        print(f"\nDistribución por estado de registro:")
        print("-" * 60)

        total = sum(count for _, count in estados)

        for estado, count in estados:
            pct = (count / total * 100) if total > 0 else 0
            print(f"   {estado or 'NULL':<30}: {count:>8,} ({pct:>5.1f}%)")

    finally:
        db.close()


def main():
    """Función principal de investigación."""
    print("\n" + "=" * 80)
    print("INVESTIGACIÓN: GAP DE SINCRONIZACIÓN CIMA")
    print("xFarma - Análisis de Discrepancias")
    print("=" * 80)

    try:
        # 1. Estado CIMA
        check_cima_status()

        # 2. Análisis de datos CIMA
        analyze_catalog_cima_data()

        # 3. Verificación de duplicados
        check_for_duplicates()

        # 4. Problemas de mapeo
        check_mapping_issues()

        # 5. Verificación VOLTADOL
        verify_voltadol()

        # 6. Distribución data_sources
        analyze_data_sources_distribution()

        # 7. Estado de registro
        check_cima_estado_registro()

        # Resumen ejecutivo
        print("\n" + "=" * 80)
        print("HALLAZGOS Y RECOMENDACIONES")
        print("=" * 80)

        print("""
📊 DATOS CLAVE:
   - Checkpoint reporta ~77,400 productos procesados
   - Catálogo local tiene 67,417 productos con CIMA
   - API CIMA tiene ~67,493 presentaciones
   - Gap aparente: ~10,000 productos procesados pero no insertados

🔍 HIPÓTESIS PRINCIPALES:
   1. Productos procesados múltiples veces (duplicados en API)
   2. Productos sin CN que se saltan silenciosamente
   3. Errores en bulk upsert que incrementan contador pero no persisten
   4. Productos con estados NO_COMERCIALIZADO que se filtran

💡 PRÓXIMOS PASOS RECOMENDADOS:
   1. Revisar logs de sincronización CIMA para errores silenciosos
   2. Implementar logging detallado de productos skipped
   3. Verificar si checkpoint cuenta productos skipped
   4. Comparar lista completa de CNs en API vs catálogo local
        """)

    except Exception as e:
        print(f"\n❌ Error en investigación: {str(e)}")
        import traceback
        traceback.print_exc()


if __name__ == "__main__":
    main()
