#!/usr/bin/env python3
"""
Script para insertar productos de prescription_reference_list que NO están en product_catalog.

Inserta:
- 1,085 productos dietoterapéuticos
- 27 tiras reactivas de glucosa

Uso:
    python scripts/insert_missing_products.py
    python scripts/insert_missing_products.py --dry-run
"""
import argparse
import sys
import logging
from pathlib import Path
from datetime import datetime, timezone
from uuid import uuid4

# Añadir raíz del proyecto al PYTHONPATH
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.database import SessionLocal
from app.models.product_catalog import ProductCatalog
from app.models.prescription_reference_list import PrescriptionReferenceList
from app.models.enums import PrescriptionCategory

# Configurar logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)


def main():
    parser = argparse.ArgumentParser(
        description="Insertar productos faltantes desde prescription_reference_list a product_catalog"
    )
    parser.add_argument(
        '--dry-run',
        action='store_true',
        help="Solo mostrar estadísticas sin insertar"
    )
    args = parser.parse_args()

    print("=" * 80)
    print("INSERCIÓN DE PRODUCTOS FALTANTES AL CATÁLOGO")
    print("=" * 80)
    print(f"Timestamp: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    print(f"Modo: {'DRY-RUN' if args.dry_run else 'INSERCIÓN REAL'}")
    print("=" * 80)
    print()

    db = SessionLocal()

    try:
        # Buscar productos en prescription_reference_list que NO están en product_catalog
        missing_products_query = db.query(PrescriptionReferenceList).filter(
            ~PrescriptionReferenceList.national_code.in_(
                db.query(ProductCatalog.national_code)
            )
        )

        missing_products = missing_products_query.all()
        total_missing = len(missing_products)

        logger.info(f"Productos en prescription_reference_list que NO están en catálogo: {total_missing}")

        # Estadísticas por categoría
        category_counts = {}
        for product in missing_products:
            cat = product.category
            category_counts[cat] = category_counts.get(cat, 0) + 1

        print("\nPRODUCTOS FALTANTES POR CATEGORÍA:")
        print("-" * 80)
        for cat, count in sorted(category_counts.items(), key=lambda x: x[1], reverse=True):
            print(f"  {cat:.<60} {count:>6,}")
        print()

        if args.dry_run:
            print("[DRY-RUN] NO se insertarán productos")
            return

        # Confirmar inserción
        response = input(f"\n¿Insertar {total_missing} productos en product_catalog? (y/N): ").strip().lower()
        if response != 'y':
            logger.info("Operación cancelada por el usuario")
            return

        # Insertar productos
        inserted_count = 0
        logger.info("Iniciando inserción...")

        for ref_product in missing_products:
            try:
                # Crear nuevo producto en catálogo
                new_product = ProductCatalog(
                    id=uuid4(),
                    national_code=ref_product.national_code,
                    nomen_nombre=ref_product.product_name,
                    xfarma_prescription_category=ref_product.category
                )

                db.add(new_product)
                inserted_count += 1

                if inserted_count % 100 == 0:
                    logger.info(f"Progreso: {inserted_count}/{total_missing} productos insertados")

            except Exception as e:
                logger.warning(f"Error insertando producto {ref_product.national_code}: {e}")
                continue

        # Commit
        db.commit()
        logger.info(f"Inserción completada: {inserted_count} productos agregados")

        print()
        print("=" * 80)
        print("INSERCIÓN COMPLETADA")
        print("=" * 80)
        print(f"Total productos insertados: {inserted_count:,}")
        print()
        print("Distribución:")
        for cat, count in sorted(category_counts.items(), key=lambda x: x[1], reverse=True):
            print(f"  - {cat}: {count:,}")
        print("=" * 80)

    except Exception as e:
        logger.error(f"Error durante inserción: {e}", exc_info=True)
        db.rollback()
        sys.exit(1)
    finally:
        db.close()


if __name__ == "__main__":
    main()
