#!/usr/bin/env python
"""
Script de inicialización de usuarios para xFarma
Crea el usuario administrador del sistema y el primer usuario de farmacia

Uso:
    python backend/scripts/init_users.py
"""

import sys
from pathlib import Path

# Añadir el directorio backend al path para imports
sys.path.append(str(Path(__file__).parent.parent))

import uuid


from app.core.security import get_password_hash
from app.database import SessionLocal
from app.models.pharmacy import Pharmacy
from app.models.user import User
from app.utils.datetime_utils import utc_now


def create_test_pharmacy(db) -> Pharmacy:
    """Crea o recupera la farmacia de prueba 'La Plaza'"""

    # Verificar si ya existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.name == "Farmacia La Plaza").first()

    if pharmacy:
        print(f"✓ Farmacia 'La Plaza' ya existe (ID: {pharmacy.id})")
        return pharmacy

    # Crear nueva farmacia
    pharmacy = Pharmacy(
        id=uuid.uuid4(),
        name="Farmacia La Plaza",
        code="FARM_PLAZA_001",
        nif="B12345678",
        license_number="LIC-2024-001",
        address="Plaza Mayor 1",
        city="Madrid",
        province="Madrid",
        postal_code="28001",
        country="España",
        phone="+34 910 123 456",
        email="info@farmaciaplaza.com",
        erp_type="farmatic",
        subscription_plan="pro",
        subscription_start=utc_now(),
        is_active=True,
        is_demo=False,
        partner_laboratories=["CINFA", "NORMON", "KERN PHARMA", "STADA", "TEVA", "SANDOZ", "MYLAN", "APOTEX"],
    )

    db.add(pharmacy)
    db.commit()
    print(f"✓ Farmacia 'La Plaza' creada (ID: {pharmacy.id})")
    return pharmacy


def create_admin_user(db, pharmacy: Pharmacy):
    """Crea el usuario administrador del sistema"""

    # Verificar si ya existe
    admin = db.query(User).filter(User.email == "dgruiz@gmail.com").first()

    if admin:
        print(f"✓ Usuario administrador ya existe (ID: {admin.id})")
        return admin

    # Crear administrador del sistema asociado a la farmacia
    admin = User(
        id=uuid.uuid4(),
        pharmacy_id=pharmacy.id,
        email="dgruiz@gmail.com",
        username="dgruiz",
        full_name="Domingo Ruiz",
        hashed_password=get_password_hash("xFarma_Admin_2024!"),
        role="admin",
        is_superuser=True,
        is_active=True,
        is_verified=True,
        email_verified_at=utc_now(),
        phone="+34 600 123 456",
        created_at=utc_now(),
        updated_at=utc_now(),
    )

    db.add(admin)
    db.commit()
    print(f"✓ Usuario administrador creado (ID: {admin.id})")
    print(f"  Email: {admin.email}")
    print("  Contraseña temporal: xFarma_Admin_2024!")
    print("  ⚠️  IMPORTANTE: Cambiar la contraseña en el primer inicio de sesión")
    return admin


def main():
    """Función principal del script"""
    print("\n" + "=" * 60)
    print("SCRIPT DE INICIALIZACIÓN DE USUARIOS - xFarma")
    print("=" * 60 + "\n")

    db = SessionLocal()

    try:
        # 1. Crear farmacia de prueba
        print("📋 Creando farmacia de prueba...")
        pharmacy = create_test_pharmacy(db)

        print("\n" + "-" * 40 + "\n")

        # 2. Crear usuario administrador del sistema (1 usuario por farmacia)
        print("👤 Creando usuario administrador del sistema...")
        admin = create_admin_user(db, pharmacy)

        print("\n" + "=" * 60)
        print("✅ INICIALIZACIÓN COMPLETADA")
        print("=" * 60 + "\n")

        # Resumen final
        print("📊 RESUMEN DE USUARIOS CREADOS:")
        print("-" * 40)

        # Contar usuarios por rol (simplificado a 2 roles - Issue #85)
        admin_count = db.query(User).filter(User.role == "admin").count()
        user_count = db.query(User).filter(User.role == "user").count()

        print(f"  Administradores: {admin_count}")
        print(f"  Usuarios: {user_count}")
        print(f"  TOTAL: {admin_count + user_count}")

        print("\n📱 CREDENCIALES DE ACCESO:")
        print("-" * 40)
        print("  ADMINISTRADOR:")
        print("    Email: dgruiz@gmail.com")
        print("    Pass: xFarma_Admin_2024!")

        print("\n⚠️  NOTA IMPORTANTE:")
        print("  - Cambiar la contraseña en el primer inicio de sesión")
        print("  - La contraseña es temporal y debe ser actualizada")
        print("  - Configurar 2FA cuando esté disponible")
        print("\n📌 RELACIÓN 1:1: Una farmacia = Un usuario (administrador)")

    except Exception as e:
        print(f"\n❌ ERROR: {e}")
        db.rollback()
        return 1

    finally:
        db.close()

    return 0


if __name__ == "__main__":
    exit(main())
