#!/usr/bin/env python3
"""
Script para inicializar/reinicializar el catálogo de productos
Útil para desarrollo y primera instalación en producción
"""

import os
import sys

sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

import argparse
import asyncio
import logging

from sqlalchemy import text

from app.database import SessionLocal
from app.services.catalog_maintenance_service import catalog_maintenance_service

# Configurar logging
logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(name)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


def clean_catalog(db):
    """
    Limpia completamente el catálogo de productos
    ADVERTENCIA: Esto eliminará TODOS los datos del catálogo
    """
    try:
        logger.warning("⚠️  LIMPIANDO CATÁLOGO COMPLETO...")

        # Limpiar tablas en orden correcto para evitar problemas de FK
        tables_to_clean = ["sales_enrichment", "homogeneous_groups", "product_catalog", "nomenclator_local"]

        for table in tables_to_clean:
            try:
                result = db.execute(text(f"DELETE FROM {table}"))
                count = result.rowcount
                db.commit()
                logger.info(f"✓ Tabla {table} limpiada: {count} registros eliminados")
            except Exception as e:
                logger.error(f"✗ Error limpiando {table}: {str(e)}")
                db.rollback()

        return True

    except Exception as e:
        logger.error(f"Error durante limpieza: {str(e)}")
        db.rollback()
        return False


async def populate_catalog(db, force=False):
    """
    Puebla el catálogo con datos frescos desde todas las fuentes
    """
    try:
        logger.info("🔄 INICIANDO POBLACIÓN DEL CATÁLOGO...")

        # Ejecutar sincronización completa
        result = await catalog_maintenance_service.full_sync(db, force_update=force)

        if result.get("status") == "success":
            logger.info("✅ CATÁLOGO POBLADO EXITOSAMENTE")

            # Mostrar estadísticas
            logger.info("\n📊 ESTADÍSTICAS DE SINCRONIZACIÓN:")

            # Nomenclator
            nomen = result.get("nomenclator", {})
            if nomen.get("status") == "updated":
                logger.info(f"  Nomenclator: {nomen.get('details', {}).get('processed', 0)} productos procesados")
            else:
                logger.info(f"  Nomenclator: {nomen.get('status', 'no data')}")

            # CIMA
            cima = result.get("cima", {})
            if cima.get("status") == "success":
                logger.info(f"  CIMA: {cima.get('total_processed', 0)} productos procesados")
                logger.info(f"    - Creados: {cima.get('created', 0)}")
                logger.info(f"    - Actualizados: {cima.get('updated', 0)}")
            else:
                logger.info(f"  CIMA: {cima.get('status', 'no data')}")

            # Catálogo
            catalog = result.get("catalog", {})
            from_nomen = catalog.get("from_nomenclator", {})
            logger.info("  Product Catalog:")
            logger.info(
                f"    - Desde Nomenclator: {from_nomen.get('created', 0)} creados, {from_nomen.get('updated', 0)} actualizados"
            )

            # Grupos homogéneos
            groups = result.get("homogeneous_groups", {})
            logger.info(f"  Grupos Homogéneos: {groups.get('groups_processed', 0)} procesados")

            # Tiempo total
            duration = result.get("duration_seconds", 0)
            logger.info(f"\n⏱️  Tiempo total: {duration:.2f} segundos")

            return True
        else:
            logger.error(f"❌ Error en población: {result.get('error', 'Unknown error')}")
            return False

    except Exception as e:
        logger.error(f"Error durante población: {str(e)}")
        return False


def get_catalog_stats(db):
    """
    Muestra estadísticas actuales del catálogo
    """
    try:
        status = catalog_maintenance_service.get_catalog_status(db)

        logger.info("\n📈 ESTADO ACTUAL DEL CATÁLOGO:")
        logger.info(f"  Estado: {status.get('status', 'unknown')}")

        catalog = status.get("catalog", {})
        logger.info(f"  Total productos: {catalog.get('total_products', 0)}")
        logger.info(f"  Con datos nomenclator: {catalog.get('with_nomenclator_data', 0)}")
        logger.info(f"  Con datos CIMA: {catalog.get('with_cima_data', 0)}")

        sync_status = catalog.get("sync_status", {})
        if sync_status:
            logger.info("  Estados de sincronización:")
            for state, count in sync_status.items():
                logger.info(f"    - {state}: {count}")

        nomen = status.get("nomenclator_local", {})
        logger.info(f"\n  Nomenclator local: {nomen.get('total_products', 0)} productos")
        logger.info(f"  Última actualización: {nomen.get('last_update', 'nunca')}")
        logger.info(f"  Datos recientes: {'Sí' if nomen.get('is_recent') else 'No'}")

        groups = status.get("homogeneous_groups", {})
        logger.info(f"\n  Grupos homogéneos: {groups.get('total_groups', 0)}")

        logger.info(f"\n  Necesita actualización: {'Sí' if status.get('needs_update') else 'No'}")

    except Exception as e:
        logger.error(f"Error obteniendo estadísticas: {str(e)}")


async def async_main():
    """Función principal async para manejar operaciones asíncronas"""
    parser = argparse.ArgumentParser(description="Inicializar catálogo de productos")
    parser.add_argument("--clean", action="store_true", help="Limpiar catálogo antes de poblar (DESTRUCTIVO)")
    parser.add_argument("--force", action="store_true", help="Forzar actualización aunque los datos sean recientes")
    parser.add_argument("--stats-only", action="store_true", help="Solo mostrar estadísticas, no actualizar")

    args = parser.parse_args()

    # Crear sesión de base de datos
    db = SessionLocal()

    try:
        if args.stats_only:
            # Solo mostrar estadísticas
            get_catalog_stats(db)
        else:
            # Limpiar si se solicita
            if args.clean:
                response = input("⚠️  ADVERTENCIA: Esto eliminará TODOS los datos del catálogo. ¿Continuar? (s/n): ")
                if response.lower() != "s":
                    logger.info("Operación cancelada")
                    return

                if not clean_catalog(db):
                    logger.error("Error durante limpieza, abortando")
                    return

            # Poblar catálogo
            if await populate_catalog(db, force=args.force):
                logger.info("\n✅ Proceso completado exitosamente")

                # Mostrar estadísticas finales
                get_catalog_stats(db)
            else:
                logger.error("\n❌ Proceso completado con errores")
                sys.exit(1)

    except KeyboardInterrupt:
        logger.info("\n\nProceso interrumpido por el usuario")
    except Exception as e:
        logger.error(f"Error fatal: {str(e)}")
        sys.exit(1)
    finally:
        db.close()


def main():
    """Función wrapper para ejecutar la función async principal"""
    asyncio.run(async_main())


if __name__ == "__main__":
    main()
