#!/usr/bin/env python3
"""
Script para asignar pharmacy_id a un usuario en Render.

PROBLEMA RESUELTO:
- Usuario sin pharmacy_id → página /generics en blanco
- Asigna farmacia respetando constraint 1:1 (REGLA #10)

USO (desde Render Dashboard → Backend Service → Shell):
    python backend/scripts/fix_user_pharmacy.py <email> <pharmacy_id>

EJEMPLO:
    python backend/scripts/fix_user_pharmacy.py dgruiz@gmail.com abc-123-def-456

UBICACIÓN: backend/scripts/ (disponible automáticamente en Render después de deploy)
"""

import sys
import os
from datetime import datetime, timezone

# Auto-detectar path del backend en diferentes entornos
def setup_backend_path():
    """Detectar y configurar el path correcto del backend."""
    possible_paths = [
        '/opt/render/project/src',  # Render production
        os.path.join(os.getcwd(), 'backend'),  # Desde root del proyecto
        os.getcwd(),  # Desde backend/
    ]

    for path in possible_paths:
        test_file = os.path.join(path, 'backend', 'app', 'database.py')
        if os.path.exists(test_file):
            backend_path = os.path.join(path, 'backend')
            sys.path.insert(0, backend_path)
            return backend_path

        # Intentar sin 'backend' intermedio
        test_file = os.path.join(path, 'app', 'database.py')
        if os.path.exists(test_file):
            sys.path.insert(0, path)
            return path

    return None

backend_path = setup_backend_path()
if not backend_path:
    print("❌ ERROR: No se pudo encontrar el módulo 'app'")
    print(f"Directorio actual: {os.getcwd()}")
    print("\n💡 Ejecuta desde el directorio raíz del proyecto:")
    print("   cd /opt/render/project/src")
    print("   python backend/scripts/fix_user_pharmacy.py <email> <pharmacy_id>")
    sys.exit(1)

from app.database import SessionLocal
from app.models import User, Pharmacy


def assign_pharmacy_to_user(email: str, pharmacy_id: str):
    """Asignar pharmacy_id a un usuario."""

    print(f"\n{'='*70}")
    print("🔧 ASIGNANDO FARMACIA A USUARIO")
    print(f"{'='*70}\n")

    db = SessionLocal()
    try:
        # 1. Verificar que el usuario existe
        user = db.query(User).filter(User.email == email).first()
        if not user:
            print(f"❌ ERROR: Usuario '{email}' NO existe\n")
            print("💡 SOLUCIÓN: Verificar el email o crear el usuario primero")
            return False

        # 2. Verificar que la farmacia existe
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            print(f"❌ ERROR: Farmacia '{pharmacy_id}' NO existe\n")
            print("💡 OPCIONES:")
            print("   1. Verificar el ID de la farmacia (copiar/pegar completo)")
            print("   2. Ejecutar diagnóstico para ver farmacias disponibles:")
            print(f"      python backend/scripts/diagnose_user.py {email}")
            return False

        # 3. Verificar que la farmacia no tiene otro usuario (constraint 1:1 - REGLA #10)
        existing_user = db.query(User).filter(User.pharmacy_id == pharmacy_id).first()
        if existing_user and existing_user.id != user.id:
            print("❌ ERROR: Constraint 1:1 violado")
            print(f"   Farmacia '{pharmacy.name}' ya tiene usuario asociado:")
            print(f"   Usuario existente: {existing_user.email}\n")
            print("💡 SOLUCIÓN:")
            print("   - Elegir otra farmacia disponible (ver diagnóstico)")
            print("   - O crear una nueva farmacia")
            return False

        # 4. Guardar estado anterior para rollback si es necesario
        old_pharmacy_id = user.pharmacy_id

        print("📋 ESTADO ACTUAL:")
        print(f"   Usuario: {user.email}")
        print(f"   pharmacy_id: {old_pharmacy_id or 'NULL'}\n")

        print("📋 CAMBIO A APLICAR:")
        print(f"   Usuario: {user.email}")
        print(f"   Farmacia: {pharmacy.name} ({pharmacy.erp_type})")
        print(f"   pharmacy_id: {pharmacy_id}\n")

        # 5. Confirmar cambio
        print("⚠️  ¿Continuar? (Ctrl+C para cancelar, Enter para continuar)")
        try:
            input()
        except KeyboardInterrupt:
            print("\n\n❌ Operación cancelada por el usuario")
            return False

        # 6. Aplicar el cambio
        user.pharmacy_id = pharmacy_id
        user.updated_at = datetime.now(timezone.utc)

        db.commit()
        db.refresh(user)

        print("\n✅ CAMBIO APLICADO EXITOSAMENTE\n")
        print("📊 VERIFICACIÓN:")
        print(f"   - email: {user.email}")
        print(f"   - pharmacy_id: {user.pharmacy_id}")
        print(f"   - farmacia: {pharmacy.name}")
        print(f"   - timestamp: {user.updated_at}\n")

        print("✨ PRÓXIMOS PASOS:")
        print("   1. Hacer login en tu app Render")
        print("   2. Navegar a /generics")
        print("   3. Verificar que los datos cargan correctamente\n")

        print("🔄 ROLLBACK (si algo sale mal):")
        if old_pharmacy_id:
            print(f"   python backend/scripts/fix_user_pharmacy.py {email} {old_pharmacy_id}")
        else:
            print("   # Restaurar a NULL:")
            print("   # Ejecutar en Python shell o crear script:")
            print("   # user.pharmacy_id = None")
            print("   # db.commit()")

        return True

    except Exception as e:
        db.rollback()
        print(f"\n❌ ERROR AL APLICAR CAMBIO: {str(e)}")
        import traceback
        traceback.print_exc()
        print("\n💡 Los cambios fueron revertidos (rollback)")
        return False
    finally:
        db.close()

    print(f"\n{'='*70}\n")


if __name__ == "__main__":
    if len(sys.argv) < 3:
        print("\n❌ ERROR: Faltan argumentos\n")
        print("USO:")
        print("   python backend/scripts/fix_user_pharmacy.py <email> <pharmacy_id>\n")
        print("EJEMPLO:")
        print("   python backend/scripts/fix_user_pharmacy.py dgruiz@gmail.com abc-123-def\n")
        print("💡 PASOS RECOMENDADOS:")
        print("   1. Ejecutar diagnóstico para ver farmacias disponibles:")
        print("      python backend/scripts/diagnose_user.py dgruiz@gmail.com")
        print("   2. Copiar el ID de una farmacia disponible")
        print("   3. Ejecutar este script con el ID copiado\n")
        sys.exit(1)

    email = sys.argv[1]
    pharmacy_id = sys.argv[2]

    success = assign_pharmacy_to_user(email, pharmacy_id)
    sys.exit(0 if success else 1)
