#!/usr/bin/env python3
"""
Script para corregir el problema de la columna NIF faltante en la tabla pharmacies.
Este script aplica las migraciones pendientes y verifica que la columna exista.
"""

import logging
import os
import sys
from pathlib import Path

from alembic import command
from alembic.config import Config
from sqlalchemy import create_engine, inspect, text

# Configurar logging
logging.basicConfig(level=logging.INFO, format="%(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


def check_column_exists(engine, table_name, column_name):
    """Verifica si una columna existe en una tabla."""
    inspector = inspect(engine)
    columns = [col["name"] for col in inspector.get_columns(table_name)]
    return column_name in columns


def apply_migrations():
    """Aplica todas las migraciones pendientes."""
    try:
        # Cambiar al directorio backend para que alembic encuentre el archivo de configuración
        backend_dir = Path(__file__).parent.parent
        os.chdir(backend_dir)

        # Configurar Alembic
        alembic_cfg = Config("alembic.ini")

        logger.info("📦 Aplicando migraciones pendientes...")
        command.upgrade(alembic_cfg, "head")
        logger.info("✅ Migraciones aplicadas exitosamente")
        return True
    except Exception as e:
        logger.error(f"❌ Error aplicando migraciones: {str(e)}")
        return False


def manual_fix_nif_column(engine):
    """Aplica un fix manual si las migraciones no funcionaron."""
    try:
        with engine.connect() as conn:
            # Verificar si la columna ya existe
            if check_column_exists(engine, "pharmacies", "nif"):
                logger.info("✅ La columna 'nif' ya existe en la tabla 'pharmacies'")
                return True

            # Añadir la columna si no existe
            logger.info("⚠️  La columna 'nif' no existe. Añadiéndola manualmente...")
            conn.execute(
                text(
                    """
                ALTER TABLE pharmacies
                ADD COLUMN IF NOT EXISTS nif VARCHAR(20)
            """
                )
            )
            conn.commit()

            logger.info("✅ Columna 'nif' añadida exitosamente")
            return True
    except Exception as e:
        logger.error(f"❌ Error añadiendo columna manualmente: {str(e)}")
        return False


def verify_all_columns(engine):
    """Verifica que todas las columnas críticas existan."""
    critical_columns = {
        "pharmacies": [
            "nif",
            "license_number",
            "province",
            "mobile",
            "website",
            "erp_type",
            "subscription_plan",
            "partner_laboratories",
        ],
        "sales_data": [
            "ean13",
            "codigo_nacional",
            "year",
            "month",
            "day",
            "purchase_price",
            "sale_price",
            "margin_percentage",
        ],
    }

    all_good = True
    for table_name, columns in critical_columns.items():
        logger.info(f"\n📋 Verificando tabla '{table_name}':")
        for column in columns:
            if check_column_exists(engine, table_name, column):
                logger.info(f"  ✅ {column}")
            else:
                logger.warning(f"  ❌ {column} - FALTANTE")
                all_good = False

    return all_good


def main():
    """Función principal del script."""
    logger.info("=" * 50)
    logger.info("   xFarma - Fix Missing NIF Column")
    logger.info("=" * 50)

    # Obtener URL de la base de datos
    database_url = os.getenv("DATABASE_URL", "postgresql://xfarma_user:xfarma_dev_2024@postgres:5432/xfarma_db")

    # Crear engine
    engine = create_engine(database_url)

    # Paso 1: Aplicar migraciones
    migrations_success = apply_migrations()

    # Paso 2: Fix manual si es necesario
    if not migrations_success:
        logger.info("\n🔧 Aplicando fix manual...")
        manual_fix_nif_column(engine)

    # Paso 3: Verificar todas las columnas críticas
    logger.info("\n🔍 Verificando estado final de la base de datos...")
    if verify_all_columns(engine):
        logger.info("\n" + "=" * 50)
        logger.info("✅ BASE DE DATOS CORREGIDA EXITOSAMENTE")
        logger.info("=" * 50)
        logger.info("\nEl panel de administración debería funcionar correctamente ahora.")
        logger.info("Accede a: http://localhost:8050")
        return 0
    else:
        logger.warning("\n" + "=" * 50)
        logger.warning("⚠️  ALGUNAS COLUMNAS SIGUEN FALTANDO")
        logger.warning("=" * 50)
        logger.warning("\nEs posible que necesites ejecutar migraciones adicionales.")
        logger.warning("Ejecuta: docker-compose exec backend alembic upgrade head")
        return 1


if __name__ == "__main__":
    sys.exit(main())
