#!/usr/bin/env python3
"""
Script de diagnóstico para ejecutar desde Render Shell.
Verifica si el usuario tiene pharmacy_id asociado.

USO (desde Render Dashboard → Backend Service → Shell):
    python scripts/diagnose_user_render_shell.py dgruiz@gmail.com
"""

import sys

# Agregar path del backend
sys.path.insert(0, '/opt/render/project/src/backend')

from app.database import SessionLocal
from app.models import User, Pharmacy


def diagnose_user(email: str):
    """Diagnosticar usuario y su pharmacy_id."""

    print(f"\n{'='*70}")
    print(f"🔍 DIAGNÓSTICO DE USUARIO: {email}")
    print(f"{'='*70}\n")

    db = SessionLocal()
    try:
        # 1. Buscar usuario
        user = db.query(User).filter(User.email == email).first()

        if not user:
            print(f"❌ ERROR: Usuario '{email}' NO existe en la base de datos\n")
            print("💡 SOLUCIÓN: Crear el usuario primero con el script de creación de usuarios")
            return

        # 2. Verificar pharmacy_id
        print("✅ Usuario encontrado:")
        print(f"   - ID: {user.id}")
        print(f"   - Email: {user.email}")
        print(f"   - Role: {user.role}")
        print(f"   - Is Active: {user.is_active}")
        print(f"   - pharmacy_id: {user.pharmacy_id or 'NULL ❌'}\n")

        if user.pharmacy_id is None:
            print("❌ PROBLEMA DETECTADO: pharmacy_id es NULL")
            print("   Esto causa que /generics muestre página en blanco\n")

            # Mostrar farmacias disponibles
            print("📋 Farmacias disponibles (sin usuario asociado):\n")

            pharmacies = db.query(Pharmacy).filter(Pharmacy.is_active == True).all()
            available = []

            for pharmacy in pharmacies:
                has_user = db.query(User).filter(User.pharmacy_id == pharmacy.id).first()
                if not has_user:
                    available.append(pharmacy)
                    print(f"   ✅ {pharmacy.id}")
                    print(f"      Nombre: {pharmacy.name}")
                    print(f"      ERP: {pharmacy.erp_type}\n")

            if not available:
                print("   ⚠️  No hay farmacias disponibles sin usuario")
                print("   💡 Deberás crear una nueva farmacia primero\n")

            print("\n🛠️  SOLUCIÓN:")
            print("   Ejecutar: python scripts/fix_user_pharmacy_render_shell.py dgruiz@gmail.com {PHARMACY_ID}")

        else:
            # Verificar que la farmacia existe
            pharmacy = db.query(Pharmacy).filter(Pharmacy.id == user.pharmacy_id).first()

            if not pharmacy:
                print(f"❌ ERROR CRÍTICO: Farmacia {user.pharmacy_id} NO existe")
                print("   El usuario tiene pharmacy_id pero la farmacia fue eliminada\n")
                print("🛠️  SOLUCIÓN:")
                print("   1. Crear farmacia con ID específico:")
                print(f"      python scripts/create_pharmacy_render_shell.py {user.pharmacy_id}")
                print("   2. O asignar otra farmacia disponible")
            else:
                print("✅ CONFIGURACIÓN CORRECTA:")
                print(f"   - Usuario: {user.email}")
                print(f"   - Farmacia: {pharmacy.name}")
                print(f"   - ERP: {pharmacy.erp_type}")
                print(f"   - Activa: {pharmacy.is_active}\n")
                print("✨ La página /generics DEBERÍA funcionar correctamente")
                print("\n💡 Si aún ves página en blanco, revisa:")
                print("   - Logs del backend (errores de autenticación)")
                print("   - Console del navegador (F12 → Console)")
                print("   - Network tab (F12 → Network)")

    except Exception as e:
        print(f"\n❌ ERROR INESPERADO: {str(e)}")
        import traceback
        traceback.print_exc()
    finally:
        db.close()

    print(f"\n{'='*70}\n")


if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("❌ ERROR: Falta el email del usuario")
        print("\nUSO:")
        print("   python scripts/diagnose_user_render_shell.py <email>")
        print("\nEJEMPLO:")
        print("   python scripts/diagnose_user_render_shell.py dgruiz@gmail.com")
        sys.exit(1)

    email = sys.argv[1]
    diagnose_user(email)
