#!/usr/bin/env python3
"""
Script de diagnóstico para verificar pharmacy_id de usuario en Render.

PROBLEMA RESUELTO:
- Usuario hace login pero /generics muestra página en blanco
- Causa: pharmacy_id es NULL en la BD

USO (desde Render Dashboard → Backend Service → Shell):
    python backend/scripts/diagnose_user.py dgruiz@gmail.com

UBICACIÓN: backend/scripts/ (disponible automáticamente en Render después de deploy)
"""

import sys
import os

# Auto-detectar path del backend en diferentes entornos
def setup_backend_path():
    """Detectar y configurar el path correcto del backend."""
    possible_paths = [
        '/opt/render/project/src',  # Render production
        os.path.join(os.getcwd(), 'backend'),  # Desde root del proyecto
        os.getcwd(),  # Desde backend/
    ]

    for path in possible_paths:
        test_file = os.path.join(path, 'backend', 'app', 'database.py')
        if os.path.exists(test_file):
            backend_path = os.path.join(path, 'backend')
            sys.path.insert(0, backend_path)
            return backend_path

        # Intentar sin 'backend' intermedio
        test_file = os.path.join(path, 'app', 'database.py')
        if os.path.exists(test_file):
            sys.path.insert(0, path)
            return path

    return None

backend_path = setup_backend_path()
if not backend_path:
    print("❌ ERROR: No se pudo encontrar el módulo 'app'")
    print(f"Directorio actual: {os.getcwd()}")
    print("\n💡 Ejecuta desde el directorio raíz del proyecto:")
    print("   cd /opt/render/project/src")
    print("   python backend/scripts/diagnose_user.py dgruiz@gmail.com")
    sys.exit(1)

from app.database import SessionLocal
from app.models import User, Pharmacy


def diagnose_user(email: str):
    """Diagnosticar usuario y su pharmacy_id."""

    print(f"\n{'='*70}")
    print(f"🔍 DIAGNÓSTICO DE USUARIO: {email}")
    print(f"{'='*70}\n")

    db = SessionLocal()
    try:
        # 1. Buscar usuario
        user = db.query(User).filter(User.email == email).first()

        if not user:
            print(f"❌ ERROR: Usuario '{email}' NO existe en la base de datos\n")
            print("💡 SOLUCIÓN: Crear el usuario primero")
            return

        # 2. Verificar pharmacy_id
        print("✅ Usuario encontrado:")
        print(f"   - ID: {user.id}")
        print(f"   - Email: {user.email}")
        print(f"   - Role: {user.role}")
        print(f"   - Is Active: {user.is_active}")
        print(f"   - pharmacy_id: {user.pharmacy_id or 'NULL ❌'}\n")

        if user.pharmacy_id is None:
            print("❌ PROBLEMA DETECTADO: pharmacy_id es NULL")
            print("   Esto causa que /generics muestre página en blanco\n")

            # Mostrar farmacias disponibles
            print("📋 Farmacias disponibles (sin usuario asociado):\n")

            pharmacies = db.query(Pharmacy).filter(Pharmacy.is_active == True).all()
            available = []

            for pharmacy in pharmacies:
                has_user = db.query(User).filter(User.pharmacy_id == pharmacy.id).first()
                if not has_user:
                    available.append(pharmacy)
                    print(f"   ✅ ID: {pharmacy.id}")
                    print(f"      Nombre: {pharmacy.name}")
                    print(f"      ERP: {pharmacy.erp_type}\n")

            if not available:
                print("   ⚠️  No hay farmacias disponibles sin usuario")
                print("   💡 Necesitarás crear una nueva farmacia primero\n")
                print("📋 Todas las farmacias existentes:\n")
                for pharmacy in pharmacies:
                    user_count = db.query(User).filter(User.pharmacy_id == pharmacy.id).count()
                    print(f"   • {pharmacy.name} ({pharmacy.erp_type})")
                    print(f"     ID: {pharmacy.id}")
                    print(f"     Usuarios asociados: {user_count}\n")

            print("\n🛠️  PRÓXIMO PASO:")
            if available:
                print(f"   python backend/scripts/fix_user_pharmacy.py {email} {{PHARMACY_ID}}")
            else:
                print("   Necesitas crear una farmacia primero o reasignar usuarios")

        else:
            # Verificar que la farmacia existe
            pharmacy = db.query(Pharmacy).filter(Pharmacy.id == user.pharmacy_id).first()

            if not pharmacy:
                print(f"❌ ERROR CRÍTICO: Farmacia {user.pharmacy_id} NO existe")
                print("   El usuario tiene pharmacy_id pero la farmacia fue eliminada\n")
                print("🛠️  SOLUCIONES:")
                print("   1. Asignar otra farmacia disponible")
                print("   2. O crear farmacia con el ID específico que falta")
            else:
                print("✅ CONFIGURACIÓN CORRECTA:")
                print(f"   - Usuario: {user.email}")
                print(f"   - Farmacia: {pharmacy.name}")
                print(f"   - ERP: {pharmacy.erp_type}")
                print(f"   - Activa: {pharmacy.is_active}\n")
                print("✨ La página /generics DEBERÍA funcionar correctamente\n")
                print("💡 Si aún ves página en blanco, revisa:")
                print("   - Logs del backend: busca errores de autenticación")
                print("   - Console del navegador: F12 → Console")
                print("   - Network tab: F12 → Network → busca requests fallidas")

    except Exception as e:
        print(f"\n❌ ERROR INESPERADO: {str(e)}")
        import traceback
        traceback.print_exc()
    finally:
        db.close()

    print(f"\n{'='*70}\n")


if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("\n❌ ERROR: Falta el email del usuario\n")
        print("USO:")
        print("   python backend/scripts/diagnose_user.py <email>\n")
        print("EJEMPLO:")
        print("   python backend/scripts/diagnose_user.py dgruiz@gmail.com\n")
        sys.exit(1)

    email = sys.argv[1]
    diagnose_user(email)
