-- Script de diagnóstico para verificar usuario en Render
-- Problema: /generics muestra página en blanco después de login
-- Causa probable: pharmacy_id faltante o NULL

-- ============================================================================
-- 1. VERIFICAR USUARIO EXISTE Y TIENE PHARMACY_ID
-- ============================================================================
SELECT
    id as user_id,
    email,
    pharmacy_id,
    role,
    is_active,
    created_at,
    CASE
        WHEN pharmacy_id IS NULL THEN '❌ PROBLEMA: pharmacy_id es NULL'
        ELSE '✅ OK: tiene pharmacy_id'
    END as diagnostico
FROM users
WHERE email = 'dgruiz@gmail.com';

-- ============================================================================
-- 2. SI pharmacy_id EXISTE, VERIFICAR QUE LA FARMACIA EXISTE
-- ============================================================================
SELECT
    p.id as pharmacy_id,
    p.name as pharmacy_name,
    p.erp_type,
    p.is_active,
    p.created_at,
    u.email as associated_user
FROM pharmacies p
LEFT JOIN users u ON u.pharmacy_id = p.id
WHERE u.email = 'dgruiz@gmail.com';

-- ============================================================================
-- 3. VERIFICAR CONSTRAINT 1:1 (User-Pharmacy)
-- ============================================================================
-- Debe haber exactamente 1 usuario por farmacia
SELECT
    pharmacy_id,
    COUNT(*) as num_users,
    string_agg(email, ', ') as users
FROM users
WHERE pharmacy_id IS NOT NULL
GROUP BY pharmacy_id
HAVING COUNT(*) > 1;
-- Si retorna filas: ❌ VIOLACIÓN de constraint 1:1 (REGLA #10)
-- Si no retorna filas: ✅ Constraint correcto

-- ============================================================================
-- 4. VERIFICAR TODOS LOS USUARIOS ADMIN (para referencia)
-- ============================================================================
SELECT
    email,
    pharmacy_id,
    role,
    is_active
FROM users
WHERE role = 'admin'
ORDER BY created_at DESC;

-- ============================================================================
-- INTERPRETACIÓN DE RESULTADOS
-- ============================================================================
/*
CASO 1: pharmacy_id es NULL
  → SOLUCIÓN: Asignar farmacia al usuario con UPDATE
  → Script: scripts/fix_missing_pharmacy_id.sql

CASO 2: pharmacy_id existe pero farmacia no existe en tabla pharmacies
  → SOLUCIÓN: Crear farmacia primero, luego asociar usuario
  → Script: scripts/create_pharmacy_for_user.sql

CASO 3: pharmacy_id existe y farmacia existe
  → PROBLEMA: Error de backend o autenticación
  → REVISAR: Logs de Render para errores 401/500
*/
