"""
Script para crear IntercambiableGroup desde clusters validados.

Issue #446: Taxonomía jerárquica de venta libre
ADR-001: docs/architecture/ADR-001-VENTA-LIBRE-TAXONOMY-HIERARCHY.md

Este script crea los primeros grupos intercambiables basados en los
clusters mixtos validados del POC (notebooks/run_unified_clustering.py).

Uso:
    # Desde Docker
    docker-compose exec backend python -m backend.scripts.create_intercambiable_groups

    # Local con DATABASE_URL
    DATABASE_URL="postgresql://..." python backend/scripts/create_intercambiable_groups.py
"""

import os
import sys
import uuid
from datetime import datetime, timezone
from typing import List, Dict

# Add parent to path for imports
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))

from sqlalchemy import create_engine, text
from sqlalchemy.orm import sessionmaker

# Database URL
DATABASE_URL = os.getenv("DATABASE_URL", "postgresql://xfarma_user:xfarma_dev_2024@localhost:5432/xfarma_db")

engine = create_engine(DATABASE_URL)
SessionLocal = sessionmaker(bind=engine)


def slugify(text: str) -> str:
    """Convierte texto a slug URL-friendly."""
    import re
    text = text.lower()
    text = re.sub(r'[áàäâ]', 'a', text)
    text = re.sub(r'[éèëê]', 'e', text)
    text = re.sub(r'[íìïî]', 'i', text)
    text = re.sub(r'[óòöô]', 'o', text)
    text = re.sub(r'[úùüû]', 'u', text)
    text = re.sub(r'[ñ]', 'n', text)
    text = re.sub(r'[^a-z0-9]+', '-', text)
    text = text.strip('-')
    return text


# =============================================================================
# GRUPOS INTERCAMBIABLES VALIDADOS (desde POC clustering)
# =============================================================================

VALIDATED_GROUPS = [
    {
        "name": "Productos para el sueño e insomnio",
        "necesidad": "insomnio",
        "subcategory": "oral",
        "description": """
Grupo de productos para conciliar el sueño y tratar el insomnio ocasional.
Incluye medicamentos OTC (doxilamina, difenhidramina) y suplementos naturales
(melatonina, valeriana, pasiflora). Todos son de venta libre y pueden
recomendarse como alternativas según preferencia del cliente.

Productos CIMA: DORMIREL, SOÑODOR, MELATONINA NOXAREM
Suplementos: Aquilea Sueño, NS Soñaben, IVB Melatonina, Mayla Sleep, Aboca Sedivitax
        """.strip(),
        "min_brands": 2,
        "min_products": 5,
        "clustering_method": "semantic",
        "example_products": [
            "DORMIREL 25 MG",
            "MELATONINA NOXAREM",
            "Aquilea Sueño Instant",
            "NS Soñaben",
        ]
    },
    {
        "name": "Vitaminas y minerales básicos",
        "necesidad": "vitaminas_minerales",
        "subcategory": "oral",
        "description": """
Grupo de suplementos vitamínicos y minerales de uso general.
Incluye medicamentos registrados (biotina, magnesio, vitaminas B)
y suplementos alimenticios equivalentes. Intercambiables para
necesidades generales de suplementación.

Productos CIMA: MEDEBIOTIN, HIDROXIL B1-B6-B12, MAGNOGENE, BECOZYME
Suplementos: Aquilea Magnesio, Aquilea Multivitaminas, IVB Vitamina D3, NS Vitans
        """.strip(),
        "min_brands": 2,
        "min_products": 5,
        "clustering_method": "semantic",
        "example_products": [
            "MEDEBIOTIN FUERTE",
            "MAGNOGENE",
            "Aquilea Magnesio",
            "IVB Vitamina D3",
        ]
    },
    {
        "name": "Probióticos y salud digestiva",
        "necesidad": "digestivo",
        "subcategory": "oral",
        "description": """
Grupo de probióticos y productos para el equilibrio de la flora intestinal.
Incluye medicamentos OTC y suplementos con lactobacilos y bifidobacterias.
Útiles para diarrea, tras antibióticos, o mantenimiento digestivo.

Productos CIMA: INFLORAN, ULTRA-LEVURA
Suplementos: Aquilea Probiomax, NS Florabiotic, Mayla Probio Balance, Aquilea Gases
        """.strip(),
        "min_brands": 2,
        "min_products": 3,
        "clustering_method": "semantic",
        "example_products": [
            "INFLORAN",
            "Aquilea Probiomax",
            "NS Florabiotic Instant",
        ]
    },
    {
        "name": "Piernas cansadas y circulación venosa",
        "necesidad": "piernas_cansadas",
        "subcategory": "oral",
        "description": """
Grupo de productos para mejorar la circulación venosa y aliviar
la sensación de piernas cansadas y pesadas. Incluye venotónicos
orales (diosmina, troxerutina) y suplementos de origen vegetal
(vid roja, castaño de indias).

Productos CIMA: VARILIV, VENORUTON, HEMOLIV
Suplementos: Aquilea Piernas Ligeras, Aquilea Piernas Gel Frío
        """.strip(),
        "min_brands": 2,
        "min_products": 3,
        "clustering_method": "semantic",
        "example_products": [
            "VARILIV DIOSMINA 500 MG",
            "VENORUTON",
            "Aquilea Piernas Ligeras",
        ]
    },
    {
        "name": "Articulaciones y dolor muscular (suplementos)",
        "necesidad": "dolor_muscular",
        "subcategory": "oral",
        "description": """
Grupo de suplementos para el cuidado de articulaciones y músculos.
Incluye analgésicos OTC orales y suplementos con colágeno,
glucosamina y condroitina. Para dolor muscular crónico y
mantenimiento articular.

Productos CIMA: ACTROMADOL (naproxeno)
Suplementos: Aquilea Articulaciones Colágeno, Aquilea Artinova, Mayla Articulaciones
        """.strip(),
        "min_brands": 2,
        "min_products": 3,
        "clustering_method": "semantic",
        "example_products": [
            "ACTROMADOL 660 MG",
            "Aquilea Articulaciones Colágeno + Magnesio",
            "Mayla Articulaciones Forte",
        ]
    },
    {
        "name": "Acidez y protección gástrica",
        "necesidad": "acidez_estomacal",
        "subcategory": "oral",
        "description": """
Grupo de productos para la acidez estomacal y protección gástrica.
Incluye antiácidos OTC (omeprazol, ranitidina, almagato) y
productos naturales con efecto protector de mucosa.

Productos CIMA: OMEPRAZOL OTC, GAVISCON, RENNIE, ALMAX
Suplementos: Aboca Neobianacid
        """.strip(),
        "min_brands": 2,
        "min_products": 3,
        "clustering_method": "semantic",
        "example_products": [
            "OMEPRAZOL 20 MG OTC",
            "GAVISCON FORTE",
            "Aboca Neobianacid",
        ]
    },
    {
        "name": "Dolor de garganta y irritación bucofaríngea",
        "necesidad": "dolor_garganta",
        "subcategory": "bucofaringea",
        "description": """
Grupo de productos para aliviar el dolor e irritación de garganta.
Incluye antisépticos bucales, antiinflamatorios locales y
productos naturales con propóleo y miel.

Productos CIMA: STREFEN, STREPSILS, LIZIPAINA
Suplementos: Aboca Golamir 2ACT Spray, Aboca Golamir Comprimidos
        """.strip(),
        "min_brands": 2,
        "min_products": 3,
        "clustering_method": "semantic",
        "example_products": [
            "STREFEN 8,75 MG",
            "STREPSILS",
            "Aboca Golamir 2ACT Spray",
        ]
    },
    {
        "name": "Tratamiento del acné facial",
        "necesidad": "acne",
        "subcategory": "facial",
        "description": """
Grupo de productos para el tratamiento del acné.
Incluye medicamentos tópicos OTC (peróxido de benzoilo)
y cosmecéuticos especializados en piel grasa y acneica.

Productos CIMA: PEROXIBEN (peróxido de benzoilo)
Cosméticos: ISDIN Acniben
        """.strip(),
        "min_brands": 2,
        "min_products": 2,
        "clustering_method": "semantic",
        "example_products": [
            "PEROXIBEN 50 MG/G GEL",
            "ISDIN Acniben Repair Gel Crema",
        ]
    },
]


def create_groups(session, dry_run: bool = False) -> List[Dict]:
    """
    Crea los grupos intercambiables en la base de datos.

    Args:
        session: Sesión SQLAlchemy
        dry_run: Si True, no hace commit (solo muestra lo que haría)

    Returns:
        Lista de grupos creados
    """
    created_groups = []

    print("\n" + "=" * 70)
    print("CREACIÓN DE GRUPOS INTERCAMBIABLES")
    print("Issue #446 - Taxonomía Venta Libre")
    print("=" * 70)

    for group_data in VALIDATED_GROUPS:
        slug = slugify(group_data["name"])

        # Verificar si ya existe
        existing = session.execute(
            text("SELECT id FROM intercambiable_group WHERE slug = :slug"),
            {"slug": slug}
        ).fetchone()

        if existing:
            print(f"\n[SKIP] {group_data['name']}")
            print(f"       Ya existe con slug: {slug}")
            continue

        group_id = str(uuid.uuid4())

        print(f"\n[CREATE] {group_data['name']}")
        print(f"         Slug: {slug}")
        print(f"         Necesidad: {group_data['necesidad']}")
        print(f"         Subcategoría: {group_data['subcategory']}")
        print(f"         Ejemplos: {', '.join(group_data['example_products'][:3])}")

        if not dry_run:
            session.execute(
                text("""
                    INSERT INTO intercambiable_group (
                        id, name, slug, description, necesidad, subcategory,
                        min_brands, min_products, clustering_method,
                        validated, validated_by, validated_at,
                        product_count, brand_count, total_sales_amount, total_sales_count,
                        created_at
                    ) VALUES (
                        :id, :name, :slug, :description, :necesidad, :subcategory,
                        :min_brands, :min_products, :clustering_method,
                        :validated, :validated_by, :validated_at,
                        0, 0, 0, 0,
                        NOW()
                    )
                """),
                {
                    "id": group_id,
                    "name": group_data["name"],
                    "slug": slug,
                    "description": group_data["description"],
                    "necesidad": group_data["necesidad"],
                    "subcategory": group_data["subcategory"],
                    "min_brands": group_data["min_brands"],
                    "min_products": group_data["min_products"],
                    "clustering_method": group_data["clustering_method"],
                    "validated": True,  # Pre-validados desde POC
                    "validated_by": "POC Clustering #446",
                    "validated_at": datetime.now(timezone.utc),
                }
            )

        created_groups.append({
            "id": group_id,
            "name": group_data["name"],
            "slug": slug,
            "necesidad": group_data["necesidad"],
        })

    if not dry_run:
        session.commit()
        print("\n[OK] Commit realizado")
    else:
        print("\n[DRY-RUN] No se hizo commit")

    return created_groups


def show_summary(session):
    """Muestra resumen de grupos creados."""
    result = session.execute(
        text("""
            SELECT
                necesidad,
                COUNT(*) as grupos,
                STRING_AGG(name, ', ') as nombres
            FROM intercambiable_group
            GROUP BY necesidad
            ORDER BY grupos DESC
        """)
    ).fetchall()

    print("\n" + "=" * 70)
    print("RESUMEN DE GRUPOS INTERCAMBIABLES")
    print("=" * 70)

    total = 0
    for row in result:
        print(f"\n{row.necesidad}: {row.grupos} grupo(s)")
        print(f"  → {row.nombres}")
        total += row.grupos

    print(f"\n{'=' * 70}")
    print(f"TOTAL: {total} grupos intercambiables")
    print("=" * 70)


def main():
    import argparse

    parser = argparse.ArgumentParser(description="Crear grupos intercambiables desde POC")
    parser.add_argument("--dry-run", action="store_true", help="No hacer commit, solo mostrar")
    args = parser.parse_args()

    session = SessionLocal()

    try:
        created = create_groups(session, dry_run=args.dry_run)

        if not args.dry_run:
            show_summary(session)

        print(f"\nGrupos procesados: {len(created)}")

    finally:
        session.close()


if __name__ == "__main__":
    main()
