#!/usr/bin/env python3
"""
Script para crear usuario administrador desde cero.

Uso:
    python backend/scripts/create_admin_user.py

O desde Docker:
    docker-compose exec backend python scripts/create_admin_user.py

Funcionalidad:
- Crea una farmacia de desarrollo si no existe
- Crea usuario admin con credenciales especificadas
- Maneja duplicados de forma segura (thread-safe)
- Valida que el usuario puede hacer login
"""

import sys
import uuid
from datetime import datetime, timezone
from pathlib import Path

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from sqlalchemy.exc import IntegrityError

from app.core.security import get_password_hash
from app.database import SessionLocal
from app.models.audit_log import AuditLog, AuditAction, AuditSeverity
from app.models.pharmacy import Pharmacy
from app.models.user import User

# Configuración del usuario admin
ADMIN_EMAIL = "dgruiz@gmail.com"
ADMIN_USERNAME = "dgruiz"
ADMIN_PASSWORD = "xFarma_Admin_2024!"
ADMIN_FULL_NAME = "Daniel Gruiz"

# Configuración de farmacia
PHARMACY_NAME = "Farmacia Principal"
PHARMACY_NIF = "B12345678"
PHARMACY_ADDRESS = "Dirección Principal"


def create_pharmacy(db: SessionLocal) -> Pharmacy:
    """
    Crea o recupera la farmacia principal.

    Args:
        db: Sesión de base de datos

    Returns:
        Pharmacy: Instancia de farmacia creada o existente
    """
    # Verificar si ya existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.nif == PHARMACY_NIF).first()

    if pharmacy:
        print(f"✓ Farmacia existente encontrada: {pharmacy.name} (ID: {pharmacy.id})")
        return pharmacy

    # Crear nueva farmacia
    pharmacy = Pharmacy(
        id=uuid.uuid4(),
        name=PHARMACY_NAME,
        nif=PHARMACY_NIF,
        address=PHARMACY_ADDRESS,
        created_at=datetime.now(timezone.utc),
        updated_at=datetime.now(timezone.utc),
    )

    try:
        db.add(pharmacy)
        db.flush()
        print(f"✓ Farmacia creada: {pharmacy.name} (ID: {pharmacy.id})")
        return pharmacy
    except IntegrityError:
        # Race condition: otra sesión la creó
        db.rollback()
        pharmacy = db.query(Pharmacy).filter(Pharmacy.nif == PHARMACY_NIF).first()
        if pharmacy:
            print(f"✓ Farmacia recuperada tras IntegrityError: {pharmacy.name}")
            return pharmacy
        raise


def create_admin_user(db: SessionLocal, pharmacy: Pharmacy) -> tuple[User, bool]:
    """
    Crea el usuario administrador si no existe.

    IMPORTANTE: NO actualiza la contraseña si el usuario ya existe.
    Esto previene resets accidentales de password (Issue: audit logging).

    Args:
        db: Sesión de base de datos
        pharmacy: Farmacia a asociar

    Returns:
        tuple[User, bool]: (Usuario, True si fue creado / False si ya existía)
    """
    # Verificar si ya existe
    user = db.query(User).filter(User.email == ADMIN_EMAIL).first()

    if user:
        print(f"⚠ Usuario existente encontrado: {user.email}")
        print("  - NO se modificará la contraseña (seguridad)")
        print(f"  - Farmacia actual: {user.pharmacy_id}")
        print(f"  - Role actual: {user.role}")
        # NO modificar password - retornar usuario existente
        return user, False

    # Crear nuevo usuario
    user = User(
        id=uuid.uuid4(),
        pharmacy_id=pharmacy.id,
        email=ADMIN_EMAIL,
        username=ADMIN_USERNAME,
        hashed_password=get_password_hash(ADMIN_PASSWORD),
        full_name=ADMIN_FULL_NAME,
        role="admin",
        is_active=True,
        is_superuser=True,
        is_verified=True,
        created_at=datetime.now(timezone.utc),
        updated_at=datetime.now(timezone.utc),
    )

    try:
        db.add(user)
        db.flush()

        # Audit logging para creación de usuario admin
        audit_log = AuditLog(
            id=uuid.uuid4(),
            timestamp=datetime.now(timezone.utc),
            user_id=user.id,
            user_email=user.email,
            user_role="admin",
            action=AuditAction.ADMIN_ACTION,
            severity=AuditSeverity.CRITICAL,
            resource_type="user",
            resource_id=str(user.id),
            method="SCRIPT",
            endpoint="/scripts/create_admin_user",
            description=f"Usuario admin creado via script: {user.email}",
            details={"script": "create_admin_user.py", "pharmacy_id": str(pharmacy.id)},
            success="success",
            created_at=datetime.now(timezone.utc),
            updated_at=datetime.now(timezone.utc),
        )
        db.add(audit_log)
        db.flush()

        print(f"✓ Usuario admin creado: {user.email} (ID: {user.id})")
        print(f"✓ Audit log registrado")
        return user, True
    except IntegrityError:
        # Race condition: otra sesión lo creó
        db.rollback()
        user = db.query(User).filter(User.email == ADMIN_EMAIL).first()
        if user:
            print(f"✓ Usuario recuperado tras IntegrityError: {user.email}")
            return user, False
        raise


def verify_user(user: User, pharmacy: Pharmacy, was_created: bool):
    """
    Imprime información de verificación del usuario.

    Args:
        user: Usuario a verificar
        pharmacy: Farmacia asociada
        was_created: True si el usuario fue creado, False si ya existía
    """
    print("\n" + "=" * 60)
    if was_created:
        print("USUARIO ADMIN CREADO EXITOSAMENTE")
        print("=" * 60)
        print(f"Email:      {user.email}")
        print(f"Username:   {user.username}")
        print(f"Password:   {ADMIN_PASSWORD}")
        print(f"Role:       {user.role}")
        print(f"Farmacia:   {pharmacy.name} (NIF: {pharmacy.nif})")
        print(f"ID Usuario: {user.id}")
        print(f"ID Farmacia: {pharmacy.id}")
        print("=" * 60)
        print("\nPuedes hacer login en http://localhost:8050 con estas credenciales")
    else:
        print("USUARIO YA EXISTE - NO SE MODIFICO")
        print("=" * 60)
        print(f"Email:      {user.email}")
        print(f"Role:       {user.role}")
        print(f"ID Usuario: {user.id}")
        print("=" * 60)
        print("\n⚠ La contraseña NO fue modificada por seguridad.")
        print("  Si necesitas resetear la contraseña, usa:")
        print("  python backend/scripts/reset_user_password.py")
    print("=" * 60 + "\n")


def main():
    """Función principal del script."""
    print("\n🚀 Iniciando creación de usuario admin...\n")

    db = SessionLocal()

    try:
        # 1. Crear/recuperar farmacia
        print("📦 Paso 1: Verificando farmacia...")
        pharmacy = create_pharmacy(db)

        # 2. Crear usuario admin (NO actualiza si ya existe)
        print("\n👤 Paso 2: Verificando usuario admin...")
        user, was_created = create_admin_user(db, pharmacy)

        # 3. Commit de cambios (solo si hubo creación)
        if was_created:
            print("\n💾 Paso 3: Guardando cambios...")
            db.commit()
            print("✓ Cambios guardados en base de datos")
        else:
            print("\n💾 Paso 3: Sin cambios que guardar (usuario ya existe)")

        # 4. Verificación
        print("\n🔍 Paso 4: Verificación final...")
        verify_user(user, pharmacy, was_created)

        return 0

    except Exception as e:
        db.rollback()
        print(f"\n❌ ERROR: {str(e)}")
        print("\nPor favor revisa los logs y contacta con soporte.")
        return 1

    finally:
        db.close()


if __name__ == "__main__":
    sys.exit(main())
