#!/usr/bin/env python3
"""
Script CLI para clasificación masiva de productos de prescripción.

Issue #16 Fase 2: Clasificación batch de ~39,502 productos del catálogo.

Uso:
    # Dry-run (solo estadísticas, sin persistir)
    python scripts/classify_catalog.py --dry-run

    # Clasificar primeros 100 productos (testing)
    python scripts/classify_catalog.py --limit 100

    # Clasificar productos con filtro específico
    python scripts/classify_catalog.py --filter nomen_estado=ALTA --limit 1000

    # Clasificar TODOS los productos (39,502)
    python scripts/classify_catalog.py --all

    # Clasificar con confirmación interactiva
    python scripts/classify_catalog.py --all --confirm
"""
import argparse
import sys
import time
import logging
from datetime import datetime
from pathlib import Path

# Añadir raíz del proyecto al PYTHONPATH
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.database import SessionLocal
from app.models.product_catalog import ProductCatalog
from app.models.prescription_reference_list import PrescriptionReferenceList
from app.services.prescription_classification_service import (
    PrescriptionClassificationService
)

# Configurar logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)


def print_banner():
    """Mostrar banner del script"""
    print("=" * 80)
    print("CLASIFICACION MASIVA DE PRODUCTOS DE PRESCRIPCION")
    print("=" * 80)
    print(f"Timestamp: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    print("Issue #16 Fase 2: xFarma Prescription Classification")
    print("=" * 80)
    print()


def print_stats(resultado, execution_time):
    """Mostrar estadísticas de clasificación"""
    print("\n" + "=" * 80)
    print("RESULTADOS DE CLASIFICACION")
    print("=" * 80)
    print(f"Total productos evaluados:      {resultado['total_products']:>10,}")
    print(f"Productos clasificados:         {resultado['classified_count']:>10,}")
    print(f"Productos OTC/parafarmacia:     {resultado['skipped_otc_count']:>10,}")
    print()
    print(f"Tasa de clasificación:          {(resultado['classified_count'] / resultado['total_products'] * 100) if resultado['total_products'] > 0 else 0:>9.2f}%")
    print(f"Tiempo de ejecución:            {execution_time:>9.2f}s")
    print()

    if resultado['category_breakdown']:
        print("DISTRIBUCION POR CATEGORIA:")
        print("-" * 80)
        for categoria, count in sorted(
            resultado['category_breakdown'].items(),
            key=lambda x: x[1],
            reverse=True
        ):
            percentage = (count / resultado['classified_count'] * 100) if resultado['classified_count'] > 0 else 0
            # Manejar tanto strings como enums
            cat_name = categoria.value if hasattr(categoria, 'value') else str(categoria)
            print(f"  {cat_name:.<60} {count:>6,} ({percentage:>5.1f}%)")

    print("=" * 80)


def confirm_action(message):
    """Pedir confirmación al usuario"""
    response = input(f"\n[!] {message} (y/N): ").strip().lower()
    return response == 'y'


def main():
    parser = argparse.ArgumentParser(
        description="Clasificar productos de prescripción en batch",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Ejemplos:
  # Dry-run con primeros 100 productos
  python scripts/classify_catalog.py --dry-run --limit 100

  # Clasificar todos los productos en estado ALTA
  python scripts/classify_catalog.py --filter nomen_estado=ALTA

  # Clasificar TODOS los 39,502 productos (requiere confirmación)
  python scripts/classify_catalog.py --all --confirm

  # Reclasificar productos que ya tienen categoría
  python scripts/classify_catalog.py --force-reclassify --limit 100
        """
    )

    parser.add_argument(
        '--dry-run',
        action='store_true',
        help="Solo mostrar estadísticas sin persistir cambios"
    )

    parser.add_argument(
        '--limit',
        type=int,
        help="Limitar clasificación a N productos (default: sin límite)"
    )

    parser.add_argument(
        '--filter',
        type=str,
        help="Filtro SQL (ejemplo: 'nomen_estado=ALTA')"
    )

    parser.add_argument(
        '--all',
        action='store_true',
        help="Clasificar TODOS los productos del catálogo"
    )

    parser.add_argument(
        '--confirm',
        action='store_true',
        help="Pedir confirmación antes de ejecutar (recomendado con --all)"
    )

    parser.add_argument(
        '--force-reclassify',
        action='store_true',
        help="Reclasificar productos que ya tienen categoría"
    )

    args = parser.parse_args()

    # Validaciones
    if not args.dry_run and not args.confirm:
        logger.warning("Ejecutando SIN confirmación. Usa --confirm para seguridad.")

    print_banner()

    # Iniciar sesión DB
    db = SessionLocal()

    try:
        # Subquery: códigos nacionales en prescription_reference_list
        reference_codes_subquery = db.query(PrescriptionReferenceList.national_code).subquery()

        # Query productos de prescripción (incluye listados oficiales)
        query = db.query(ProductCatalog).filter(
            (ProductCatalog.nomen_codigo_homogeneo.isnot(None)) |
            (ProductCatalog.cima_requiere_receta == True) |
            (ProductCatalog.national_code.in_(reference_codes_subquery))  # NUEVO
        )

        # Aplicar filtro si se especifica
        if args.filter:
            field, value = args.filter.split('=')
            if hasattr(ProductCatalog, field):
                query = query.filter(getattr(ProductCatalog, field) == value)
                logger.info(f"Aplicando filtro: {field}={value}")
            else:
                logger.error(f"Campo '{field}' no existe en ProductCatalog")
                sys.exit(1)

        # Filtrar solo no clasificados (si no es reclasificación forzada)
        if not args.force_reclassify:
            query = query.filter(ProductCatalog.xfarma_prescription_category.is_(None))

        # Obtener total ANTES de aplicar límite
        total_available = query.count()

        # Aplicar límite
        if args.limit:
            query = query.limit(args.limit)
            logger.info(f"Limitando a {args.limit} productos de {total_available} disponibles")
        elif args.all:
            logger.info(f"Clasificando TODOS los {total_available} productos disponibles")
        else:
            # Default: primeros 1000 sin confirmación
            query = query.limit(1000)
            logger.info(f"Clasificando primeros 1000 productos de {total_available} disponibles")

        productos = query.all()

        logger.info(f"Productos a clasificar: {len(productos)}")

        # Confirmación interactiva
        if args.confirm and not args.dry_run:
            if not confirm_action(
                f"¿Clasificar {len(productos)} productos y persistir en BD?"
            ):
                logger.info("Operación cancelada por el usuario")
                sys.exit(0)

        # Ejecutar clasificación
        logger.info("Iniciando clasificación...")
        start_time = time.time()

        service = PrescriptionClassificationService(db=db)
        resultado = service.bulk_classify(productos, dry_run=args.dry_run)

        execution_time = time.time() - start_time

        # Mostrar resultados
        print_stats(resultado, execution_time)

        # Validar tasa de clasificación
        classification_rate = (
            (resultado['classified_count'] / resultado['total_products'] * 100)
            if resultado['total_products'] > 0 else 0
        )

        if not args.dry_run:
            if classification_rate >= 95.0:
                print(f"\n[OK] EXITO: Tasa de clasificacion >= 95%")
                print(f"Issue #16 Fase 2 COMPLETADO ({classification_rate:.2f}%)")
            elif classification_rate >= 80.0:
                print(f"\n[!] ADVERTENCIA: Tasa de clasificacion {classification_rate:.2f}% (objetivo: >=95%)")
            else:
                print(f"\n[ERROR] Tasa de clasificacion {classification_rate:.2f}% muy baja (objetivo: >=95%)")
                sys.exit(1)
        else:
            print("\n[DRY-RUN] Cambios NO fueron persistidos")

    except Exception as e:
        logger.error(f"Error durante clasificación: {e}", exc_info=True)
        sys.exit(1)
    finally:
        db.close()


if __name__ == "__main__":
    main()
