#!/usr/bin/env python3
"""
Script para verificar productos de farmacia Farmatic en manual_review.

USO (desde Render Dashboard → Backend Service → Shell):
    python backend/scripts/check_farmatic_products.py fes@fes.es

O para ver todos los productos sin codigo_nacional:
    python backend/scripts/check_farmatic_products.py fes@fes.es --no-cn

Issue #447: Investigar cómo Farmatic maneja productos sin EAN.
"""

import sys
import os

# Agregar path del backend
sys.path.insert(0, '/opt/render/project/src/backend')
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from sqlalchemy import text


def check_farmatic_products(email: str, show_no_cn: bool = False):
    """Verificar productos de farmacia Farmatic en manual_review."""

    from app.database import SessionLocal

    print(f"\n{'='*70}")
    print(f"🔍 ANÁLISIS PRODUCTOS MANUAL_REVIEW - {email}")
    print(f"{'='*70}\n")

    db = SessionLocal()
    try:
        # Verificar que el usuario existe y tiene farmacia
        user_check = db.execute(text("""
            SELECT u.id, u.email, p.id as pharmacy_id, p.name, p.erp_type
            FROM users u
            JOIN pharmacies p ON u.pharmacy_id = p.id
            WHERE u.email = :email
        """), {"email": email}).fetchone()

        if not user_check:
            print(f"❌ Usuario '{email}' no encontrado o sin farmacia asociada")
            return

        print(f"✅ Usuario encontrado:")
        print(f"   - Email: {user_check.email}")
        print(f"   - Farmacia: {user_check.name}")
        print(f"   - ERP: {user_check.erp_type}")
        print(f"   - Pharmacy ID: {user_check.pharmacy_id}\n")

        # Estadísticas generales de manual_review
        stats = db.execute(text("""
            SELECT
                COUNT(*) as total_products,
                SUM(CASE WHEN sd.codigo_nacional IS NOT NULL AND sd.codigo_nacional != '' THEN 1 ELSE 0 END) as with_cn,
                SUM(CASE WHEN sd.ean13 IS NOT NULL AND sd.ean13 != '' THEN 1 ELSE 0 END) as with_ean,
                SUM(CASE WHEN (sd.codigo_nacional IS NULL OR sd.codigo_nacional = '')
                         AND (sd.ean13 IS NULL OR sd.ean13 = '') THEN 1 ELSE 0 END) as no_codes
            FROM sales_data sd
            JOIN sales_enrichment se ON sd.id = se.sales_data_id
            WHERE sd.pharmacy_id = :pharmacy_id
              AND se.enrichment_status = 'manual_review'
        """), {"pharmacy_id": user_check.pharmacy_id}).fetchone()

        print(f"📊 ESTADÍSTICAS MANUAL_REVIEW:")
        print(f"   - Total productos: {stats.total_products}")
        print(f"   - Con codigo_nacional: {stats.with_cn}")
        print(f"   - Con EAN: {stats.with_ean}")
        print(f"   - Sin ningún código: {stats.no_codes}\n")

        if show_no_cn:
            # Mostrar productos SIN codigo_nacional
            query = """
                SELECT
                    sd.codigo_nacional,
                    sd.ean13,
                    LEFT(sd.product_name, 50) as name,
                    COUNT(*) as ventas,
                    SUM(sd.units) as units,
                    ROUND(SUM(sd.pvp_total)::numeric, 2) as importe
                FROM sales_data sd
                JOIN sales_enrichment se ON sd.id = se.sales_data_id
                WHERE sd.pharmacy_id = :pharmacy_id
                  AND se.enrichment_status = 'manual_review'
                  AND (sd.codigo_nacional IS NULL OR sd.codigo_nacional = '')
                GROUP BY sd.codigo_nacional, sd.ean13, sd.product_name
                ORDER BY ventas DESC
                LIMIT 30
            """
            title = "PRODUCTOS SIN CODIGO_NACIONAL"
        else:
            # Mostrar productos con GH o patrones interesantes
            query = """
                SELECT
                    sd.codigo_nacional,
                    sd.ean13,
                    LEFT(sd.product_name, 50) as name,
                    COUNT(*) as ventas,
                    SUM(sd.units) as units,
                    ROUND(SUM(sd.pvp_total)::numeric, 2) as importe
                FROM sales_data sd
                JOIN sales_enrichment se ON sd.id = se.sales_data_id
                WHERE sd.pharmacy_id = :pharmacy_id
                  AND se.enrichment_status = 'manual_review'
                  AND (
                      sd.product_name ILIKE '%GH %'
                      OR sd.product_name ILIKE '%GERMINAL%'
                      OR sd.product_name ILIKE '%ISDIN%'
                      OR sd.product_name ILIKE '%VICHY%'
                      OR sd.product_name ILIKE '%ROCHE%'
                      OR (sd.codigo_nacional IS NULL OR sd.codigo_nacional = '')
                  )
                GROUP BY sd.codigo_nacional, sd.ean13, sd.product_name
                ORDER BY ventas DESC
                LIMIT 30
            """
            title = "PRODUCTOS PARAFARMACIA / SIN CN"

        products = db.execute(text(query), {"pharmacy_id": user_check.pharmacy_id}).fetchall()

        print(f"\n📋 {title}:")
        print("-" * 90)
        print(f"{'CN':<12} {'EAN':<15} {'NOMBRE':<50} {'VENTAS':>8}")
        print("-" * 90)

        for p in products:
            cn = p.codigo_nacional or "(vacío)"
            ean = p.ean13 or "(vacío)"
            print(f"{cn:<12} {ean:<15} {p.name:<50} {p.ventas:>8}")

        print("-" * 90)
        print(f"Total mostrados: {len(products)}\n")

        # Análisis de patrones de código
        print("\n📈 ANÁLISIS DE PATRONES DE CÓDIGO:")
        print("-" * 50)

        patterns = db.execute(text("""
            SELECT
                CASE
                    WHEN sd.codigo_nacional IS NULL OR sd.codigo_nacional = '' THEN 'SIN_CN'
                    WHEN LENGTH(sd.codigo_nacional) = 6 THEN 'CN_6_DIGITS'
                    WHEN LENGTH(sd.codigo_nacional) = 7 THEN 'CN_7_DIGITS'
                    WHEN sd.codigo_nacional ~ '^[0-9]+$' THEN 'CN_NUMERICO'
                    ELSE 'CN_OTRO'
                END as cn_pattern,
                CASE
                    WHEN sd.ean13 IS NULL OR sd.ean13 = '' THEN 'SIN_EAN'
                    WHEN sd.ean13 LIKE '847000%' THEN 'EAN_847000_FARMACIA'
                    WHEN sd.ean13 LIKE '843%' THEN 'EAN_843_GENERAL'
                    WHEN sd.ean13 LIKE '84%' THEN 'EAN_84_ESPAÑA'
                    ELSE 'EAN_OTRO'
                END as ean_pattern,
                COUNT(*) as count
            FROM sales_data sd
            JOIN sales_enrichment se ON sd.id = se.sales_data_id
            WHERE sd.pharmacy_id = :pharmacy_id
              AND se.enrichment_status = 'manual_review'
            GROUP BY cn_pattern, ean_pattern
            ORDER BY count DESC
        """), {"pharmacy_id": user_check.pharmacy_id}).fetchall()

        for pat in patterns:
            print(f"   {pat.cn_pattern:<15} + {pat.ean_pattern:<20} = {pat.count:>6} productos")

        print()

    except Exception as e:
        print(f"\n❌ ERROR: {str(e)}")
        import traceback
        traceback.print_exc()
    finally:
        db.close()

    print(f"{'='*70}\n")


if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("❌ ERROR: Falta el email del usuario")
        print("\nUSO:")
        print("   python backend/scripts/check_farmatic_products.py <email>")
        print("   python backend/scripts/check_farmatic_products.py <email> --no-cn")
        print("\nEJEMPLO:")
        print("   python backend/scripts/check_farmatic_products.py fes@fes.es")
        sys.exit(1)

    email = sys.argv[1]
    show_no_cn = "--no-cn" in sys.argv

    check_farmatic_products(email, show_no_cn)
