#!/usr/bin/env python
# backend/scripts/backfill_prescription_categories.py
"""
Script de Backfill para xfarma_prescription_category (Issue #451)

Corrige productos con cima_requiere_receta=TRUE pero xfarma_prescription_category=NULL

Uso:
    # Dry run (validación sin cambios)
    python backend/scripts/backfill_prescription_categories.py --dry-run

    # Validar muestra de 100 productos
    python backend/scripts/backfill_prescription_categories.py --validate

    # Ejecutar backfill completo
    python backend/scripts/backfill_prescription_categories.py

    # Backfill solo productos con receta
    python backend/scripts/backfill_prescription_categories.py --requires-prescription

    # Backfill códigos específicos
    python backend/scripts/backfill_prescription_categories.py --codes 895805 929653 722396

    # Limitar a N productos
    python backend/scripts/backfill_prescription_categories.py --limit 1000 --dry-run
"""

import argparse
import logging
import sys
from pathlib import Path

# Añadir backend al path para imports
backend_path = Path(__file__).resolve().parent.parent
sys.path.insert(0, str(backend_path))

from app.database import SessionLocal
from app.services.prescription_category_backfill_service import PrescriptionCategoryBackfillService

# Configurar logging
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s - %(name)s - %(levelname)s - %(message)s"
)
logger = logging.getLogger(__name__)


def main():
    """Función principal del script"""
    parser = argparse.ArgumentParser(
        description="Backfill de xfarma_prescription_category para productos con receta (Issue #451)"
    )

    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="Ejecutar en modo dry-run (no persiste cambios, solo estadísticas)"
    )

    parser.add_argument(
        "--validate",
        action="store_true",
        help="Validar clasificación automática con muestra de 100 productos"
    )

    parser.add_argument(
        "--batch-size",
        type=int,
        default=500,
        help="Tamaño de batch para procesamiento (default: 500)"
    )

    parser.add_argument(
        "--requires-prescription",
        action="store_true",
        help="Solo procesar productos con cima_requiere_receta=TRUE"
    )

    parser.add_argument(
        "--has-nomenclator",
        action="store_true",
        help="Solo procesar productos con datos de nomenclator"
    )

    parser.add_argument(
        "--limit",
        type=int,
        help="Limitar número máximo de productos a procesar"
    )

    parser.add_argument(
        "--codes",
        nargs="+",
        help="Lista de códigos nacionales específicos a procesar"
    )

    args = parser.parse_args()

    # Banner
    logger.info("=" * 80)
    logger.info("BACKFILL DE xfarma_prescription_category - Issue #451")
    logger.info("=" * 80)
    logger.info("")

    if args.dry_run:
        logger.warning("⚠️  MODO DRY-RUN: No se persistirán cambios en la base de datos")
        logger.info("")

    # Crear sesión de base de datos
    db = SessionLocal()

    try:
        # Crear servicio de backfill
        backfill_service = PrescriptionCategoryBackfillService(db=db)

        # Modo validación
        if args.validate:
            logger.info("🔍 MODO VALIDACIÓN: Verificando clasificación automática...")
            logger.info("")

            stats = backfill_service.validate_classification(limit=100)

            logger.info("")
            logger.info("=" * 80)
            logger.info("RESULTADOS DE VALIDACIÓN")
            logger.info("=" * 80)
            logger.info(f"Total evaluado: {stats['total_evaluated']}")
            logger.info(f"Total clasificado: {stats['total_classified']}")
            logger.info(f"Total OTC (omitido): {stats['total_skipped_otc']}")
            logger.info("")
            logger.info("Distribución por categoría:")
            for category, count in sorted(
                stats["category_breakdown"].items(),
                key=lambda x: x[1],
                reverse=True
            ):
                percentage = (count / stats["total_classified"] * 100) if stats["total_classified"] > 0 else 0
                logger.info(f"  - {category}: {count} ({percentage:.1f}%)")

            return

        # Backfill por códigos específicos
        if args.codes:
            logger.info(f"🎯 Procesando {len(args.codes)} códigos nacionales específicos")
            logger.info(f"Códigos: {', '.join(args.codes)}")
            logger.info("")

            stats = backfill_service.backfill_by_national_codes(
                national_codes=args.codes,
                dry_run=args.dry_run
            )

        # Backfill con filtros
        else:
            # Construir filtros
            filter_criteria = {}

            if args.requires_prescription:
                filter_criteria["requires_prescription"] = True
                logger.info("📋 Filtro: Solo productos con cima_requiere_receta=TRUE")

            if args.has_nomenclator:
                filter_criteria["has_nomenclator"] = True
                logger.info("📋 Filtro: Solo productos con datos de nomenclator")

            if args.limit:
                filter_criteria["limit"] = args.limit
                logger.info(f"📋 Filtro: Límite de {args.limit} productos")

            logger.info("")

            # Ejecutar backfill
            stats = backfill_service.backfill_missing_categories(
                batch_size=args.batch_size,
                dry_run=args.dry_run,
                filter_criteria=filter_criteria if filter_criteria else None
            )

        # Mostrar resumen final
        logger.info("")
        logger.info("=" * 80)
        logger.info("RESUMEN DE BACKFILL")
        logger.info("=" * 80)
        logger.info(f"Total evaluado: {stats['total_evaluated']}")
        logger.info(f"Total clasificado: {stats['total_classified']}")
        logger.info(f"Total OTC (omitido): {stats['total_skipped_otc']}")
        logger.info(f"Batches procesados: {stats['batch_count']}")
        logger.info(f"Modo: {'DRY-RUN' if stats['dry_run'] else 'PRODUCCIÓN'}")
        logger.info("")

        if stats["category_breakdown"]:
            logger.info("Distribución por categoría:")
            for category, count in sorted(
                stats["category_breakdown"].items(),
                key=lambda x: x[1],
                reverse=True
            ):
                percentage = (count / stats["total_classified"] * 100) if stats["total_classified"] > 0 else 0
                logger.info(f"  - {category}: {count} ({percentage:.1f}%)")

        logger.info("")
        logger.info("=" * 80)

        if args.dry_run:
            logger.warning("⚠️  DRY-RUN completado - No se realizaron cambios en la base de datos")
            logger.info("Ejecuta sin --dry-run para aplicar los cambios")
        else:
            logger.info("✅ Backfill completado exitosamente")

        logger.info("=" * 80)

    except Exception as e:
        logger.error(f"❌ Error durante backfill: {e}", exc_info=True)
        db.rollback()
        sys.exit(1)

    finally:
        db.close()


if __name__ == "__main__":
    main()
