#!/usr/bin/env python3
"""
Script standalone para backfill de códigos ATC.

Diseñado para ejecutarse como Render Cron Job o manualmente desde línea de comandos.

Uso:
    # Backfill completo (todos los productos sin ATC)
    python backend/scripts/backfill_atc_codes.py

    # Backfill incremental (solo productos de última semana)
    python backend/scripts/backfill_atc_codes.py --incremental

    # Testing con límite
    python backend/scripts/backfill_atc_codes.py --max-products 100

Issue relacionado: #400 - Dashboard de Análisis de Ventas de Prescripción
Prerequisito: Códigos ATC necesarios para análisis terapéutico
"""

import argparse
import asyncio
import logging
import os
import sys

# Agregar directorio padre al path para imports
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from sqlalchemy import text

from app.database import SessionLocal
from app.services.atc_backfill_service import atc_backfill_service

# Configurar logging
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
    handlers=[
        logging.StreamHandler(sys.stdout),
        logging.FileHandler("atc_backfill.log") if not os.getenv("RENDER") else logging.StreamHandler(sys.stdout)
    ]
)
logger = logging.getLogger(__name__)


def parse_args():
    """Parsea argumentos de línea de comandos"""
    parser = argparse.ArgumentParser(
        description="Backfill de códigos ATC desde API de CIMA"
    )

    parser.add_argument(
        "--batch-size",
        type=int,
        default=100,
        help="Número de productos a procesar por batch (default: 100)"
    )

    parser.add_argument(
        "--concurrent-requests",
        type=int,
        default=5,
        help="Número de requests HTTP concurrentes (default: 5, max: 10)"
    )

    parser.add_argument(
        "--incremental",
        action="store_true",
        help="Solo procesar productos agregados en última semana"
    )

    parser.add_argument(
        "--max-products",
        type=int,
        default=None,
        help="Límite máximo de productos a procesar (para testing)"
    )

    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="Modo dry-run: simula ejecución sin modificar BD"
    )

    return parser.parse_args()


async def main():
    """Función principal del script"""
    args = parse_args()

    logger.info("=" * 80)
    logger.info("BACKFILL DE CÓDIGOS ATC - INICIO")
    logger.info("=" * 80)
    logger.info(f"Parámetros:")
    logger.info(f"  - batch_size: {args.batch_size}")
    logger.info(f"  - concurrent_requests: {args.concurrent_requests}")
    logger.info(f"  - incremental: {args.incremental}")
    logger.info(f"  - max_products: {args.max_products}")
    logger.info(f"  - dry_run: {args.dry_run}")
    logger.info("=" * 80)

    if args.dry_run:
        logger.warning("[ADVERTENCIA] MODO DRY-RUN: No se modificara la base de datos")

    try:
        # Validar conexión a BD
        logger.info("Verificando conexión a base de datos...")
        db = SessionLocal()
        db.execute(text("SELECT 1"))
        db.close()
        logger.info("[OK] Conexion a BD exitosa")

        if args.dry_run:
            logger.info("[OK] Dry-run completado - Sin modificaciones a BD")
            return 0

        # Ejecutar backfill
        logger.info("\nIniciando backfill de códigos ATC...")
        result = await atc_backfill_service.backfill_atc_codes(
            batch_size=args.batch_size,
            concurrent_requests=args.concurrent_requests,
            incremental=args.incremental,
            max_products=args.max_products
        )

        # Mostrar resultado
        logger.info("=" * 80)
        logger.info("BACKFILL DE CÓDIGOS ATC - RESULTADO")
        logger.info("=" * 80)
        logger.info(f"Status: {result['status']}")
        logger.info(f"Productos procesados: {result['processed']}")
        logger.info(f"Exitosos: {result['successful']}")
        logger.info(f"Fallidos: {result['failed']}")
        logger.info(f"Omitidos: {result['skipped']}")
        logger.info(f"Duración: {result['duration_seconds']:.2f} segundos")
        logger.info("")
        logger.info("Cobertura ATC:")
        coverage = result['coverage']
        logger.info(f"  - Total productos CIMA: {coverage['total_cima_products']}")
        logger.info(f"  - Con códigos ATC: {coverage['products_with_atc']}")
        logger.info(f"  - Cobertura: {coverage['coverage_percentage']:.2f}%")
        logger.info(f"  - Target: {coverage['target_percentage']}%")
        logger.info(f"  - Target alcanzado: {'SI' if coverage['target_reached'] else 'NO'}")
        logger.info("=" * 80)

        if coverage['target_reached']:
            logger.info("[OK] Issue #400 DESBLOQUEADO - Cobertura ATC suficiente")
        else:
            remaining = coverage['target_percentage'] - coverage['coverage_percentage']
            logger.warning(f"[ADVERTENCIA] Falta {remaining:.2f}% para alcanzar target de Issue #400")

        return 0 if result['status'] == 'completed' else 1

    except KeyboardInterrupt:
        logger.warning("\n[ADVERTENCIA] Backfill interrumpido por usuario")
        return 130  # Exit code estándar para SIGINT
    except Exception as e:
        logger.error(f"[ERROR] Error ejecutando backfill: {str(e)}", exc_info=True)
        return 1


if __name__ == "__main__":
    exit_code = asyncio.run(main())
    sys.exit(exit_code)
