"""
Script para asignar productos a grupos intercambiables.

Issue #446: Taxonomía jerárquica de venta libre

Este script asigna productos venta_libre a los grupos intercambiables
existentes basándose en la coincidencia de ml_category (necesidad).

Uso:
    # Dry-run (ver qué haría)
    python assign_products_to_groups.py --dry-run

    # Ejecutar asignación
    python assign_products_to_groups.py

    # Solo un grupo específico
    python assign_products_to_groups.py --group-slug "productos-para-el-sueno-e-insomnio"
"""

import os
import sys
import argparse

# Add parent to path
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))

from sqlalchemy import create_engine, text
from sqlalchemy.orm import sessionmaker

# Database URL
DATABASE_URL = os.getenv(
    "DATABASE_URL",
    "postgresql://xfarma_user:xfarma_dev_2024@localhost:5432/xfarma_db"
)

engine = create_engine(DATABASE_URL)
SessionLocal = sessionmaker(bind=engine)


def get_groups_with_potential(session):
    """Obtiene grupos con cantidad de productos potenciales a asignar."""
    result = session.execute(text("""
        SELECT
            ig.id,
            ig.name,
            ig.slug,
            ig.necesidad,
            ig.subcategory,
            ig.product_count as current_products,
            COUNT(DISTINCT se.id) as potential_products
        FROM intercambiable_group ig
        LEFT JOIN sales_enrichment se ON (
            se.ml_category = ig.necesidad
            AND se.product_type = 'venta_libre'
            AND se.intercambiable_group_id IS NULL
            AND (ig.subcategory IS NULL OR se.ml_subcategory = ig.subcategory)
        )
        GROUP BY ig.id, ig.name, ig.slug, ig.necesidad, ig.subcategory, ig.product_count
        ORDER BY potential_products DESC
    """)).fetchall()

    return result


def assign_group(session, group_slug: str, dry_run: bool = False):
    """Asigna productos a un grupo específico."""
    # Obtener grupo
    group = session.execute(
        text("SELECT id, name, necesidad, subcategory FROM intercambiable_group WHERE slug = :slug"),
        {"slug": group_slug}
    ).fetchone()

    if not group:
        print(f"[ERROR] Grupo no encontrado: {group_slug}")
        return 0

    print(f"\n[GROUP] {group.name}")
    print(f"        Necesidad: {group.necesidad}")
    print(f"        Subcategory: {group.subcategory or '(todas)'}")

    # Contar productos potenciales
    count_query = """
        SELECT COUNT(*) FROM sales_enrichment se
        WHERE se.ml_category = :necesidad
        AND se.product_type = 'venta_libre'
        AND se.intercambiable_group_id IS NULL
    """
    params = {"necesidad": group.necesidad}

    if group.subcategory:
        count_query += " AND se.ml_subcategory = :subcategory"
        params["subcategory"] = group.subcategory

    potential = session.execute(text(count_query), params).scalar()

    print(f"        Productos potenciales: {potential}")

    if dry_run:
        print(f"        [DRY-RUN] Se asignarían {potential} productos")
        return potential

    # Ejecutar asignación
    update_query = """
        UPDATE sales_enrichment
        SET intercambiable_group_id = :group_id
        WHERE ml_category = :necesidad
        AND product_type = 'venta_libre'
        AND intercambiable_group_id IS NULL
    """
    params["group_id"] = str(group.id)

    if group.subcategory:
        update_query += " AND ml_subcategory = :subcategory"

    result = session.execute(text(update_query), params)
    assigned = result.rowcount

    print(f"        [OK] Asignados: {assigned} productos")

    # Actualizar estadísticas del grupo
    session.execute(text("""
        UPDATE intercambiable_group
        SET
            product_count = (
                SELECT COUNT(DISTINCT sd.product_name)
                FROM sales_enrichment se
                JOIN sales_data sd ON se.sales_data_id = sd.id
                WHERE se.intercambiable_group_id = :group_id
            ),
            brand_count = (
                SELECT COUNT(DISTINCT se.detected_brand)
                FROM sales_enrichment se
                WHERE se.intercambiable_group_id = :group_id
                AND se.detected_brand IS NOT NULL
            ),
            total_sales_amount = (
                SELECT COALESCE(SUM(sd.total_amount), 0)
                FROM sales_enrichment se
                JOIN sales_data sd ON se.sales_data_id = sd.id
                WHERE se.intercambiable_group_id = :group_id
            ),
            total_sales_count = (
                SELECT COUNT(*)
                FROM sales_enrichment se
                WHERE se.intercambiable_group_id = :group_id
            ),
            updated_at = NOW()
        WHERE id = :group_id
    """), {"group_id": str(group.id)})

    session.commit()

    return assigned


def main():
    parser = argparse.ArgumentParser(description="Asignar productos a grupos intercambiables")
    parser.add_argument("--dry-run", action="store_true", help="No hacer cambios, solo mostrar")
    parser.add_argument("--group-slug", type=str, help="Solo procesar este grupo")
    args = parser.parse_args()

    session = SessionLocal()

    try:
        print("=" * 70)
        print("ASIGNACION DE PRODUCTOS A GRUPOS INTERCAMBIABLES")
        print("Issue #446 - Taxonomia Venta Libre")
        print("=" * 70)

        if args.group_slug:
            # Solo un grupo
            assigned = assign_group(session, args.group_slug, args.dry_run)
            print(f"\nTotal asignados: {assigned}")
        else:
            # Todos los grupos
            groups = get_groups_with_potential(session)

            print(f"\nGrupos encontrados: {len(groups)}")
            print("-" * 70)

            total_assigned = 0
            for g in groups:
                assigned = assign_group(session, g.slug, args.dry_run)
                total_assigned += assigned

            print("\n" + "=" * 70)
            print(f"TOTAL ASIGNADOS: {total_assigned}")
            print("=" * 70)

        # Mostrar resumen final
        if not args.dry_run:
            print("\n[RESUMEN FINAL]")
            result = session.execute(text("""
                SELECT
                    ig.name,
                    ig.necesidad,
                    ig.product_count,
                    ig.brand_count,
                    ig.total_sales_amount
                FROM intercambiable_group ig
                ORDER BY ig.total_sales_amount DESC
            """)).fetchall()

            for r in result:
                print(f"  {r.name}")
                print(f"    Productos: {r.product_count}, Marcas: {r.brand_count}, Ventas: {r.total_sales_amount:.2f} EUR")

    finally:
        session.close()


if __name__ == "__main__":
    main()
