"""
Script de análisis: Gap de productos de venta libre sin cobertura CIMA/Nomenclator

Este script analiza los datos existentes para:
1. Cuantificar ventas sin enriquecer (manual_review, failed)
2. Identificar productos con CN que NO están en CIMA/Nomenclator (candidatos BOT)
3. Identificar productos sin CN (códigos internos puros)
4. Simular un catálogo crowdsourced con los datos actuales

Ejecutar:
    docker-compose exec backend python backend/scripts/analyze_free_sale_gap.py

    O desde local con .env:
    python backend/scripts/analyze_free_sale_gap.py
"""

import os
import sys
from collections import defaultdict
from decimal import Decimal

# Añadir path del proyecto
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from sqlalchemy import func, and_, or_, text
from sqlalchemy.orm import Session

from backend.app.database import SessionLocal, engine
from backend.app.models.sales_data import SalesData
from backend.app.models.sales_enrichment import SalesEnrichment
from backend.app.models.product_catalog import ProductCatalog
from backend.app.models.pharmacy import Pharmacy


def format_currency(value):
    """Formatea valor como moneda EUR"""
    if value is None:
        return "€0.00"
    return f"€{float(value):,.2f}"


def format_percent(value, total):
    """Formatea porcentaje"""
    if total == 0:
        return "0.0%"
    return f"{(value / total * 100):.1f}%"


def analyze_enrichment_status(db: Session):
    """Analiza el estado de enriquecimiento de todas las ventas."""
    print("\n" + "=" * 80)
    print("1. ANÁLISIS DE ESTADO DE ENRIQUECIMIENTO")
    print("=" * 80)

    # Total de ventas
    total_sales = db.query(func.count(SalesData.id)).scalar()
    print(f"\nTotal registros de ventas: {total_sales:,}")

    # Ventas con enriquecimiento
    total_enrichments = db.query(func.count(SalesEnrichment.id)).scalar()
    print(f"Registros con enriquecimiento: {total_enrichments:,}")

    # Desglose por estado
    status_counts = db.query(
        SalesEnrichment.enrichment_status,
        func.count(SalesEnrichment.id)
    ).group_by(SalesEnrichment.enrichment_status).all()

    print(f"\nDesglose por estado de enriquecimiento:")
    print("-" * 50)

    status_dict = {}
    for status, count in status_counts:
        status_dict[status] = count
        pct = format_percent(count, total_enrichments) if total_enrichments > 0 else "0%"
        print(f"  {status or 'NULL':<20}: {count:>8,} ({pct})")

    # Calcular gap
    enriched = status_dict.get('enriched', 0)
    manual_review = status_dict.get('manual_review', 0)
    failed = status_dict.get('failed', 0)
    pending = status_dict.get('pending', 0)

    gap = manual_review + failed + pending
    print(f"\n📊 RESUMEN:")
    print(f"  ✅ Enriquecidos correctamente: {enriched:,} ({format_percent(enriched, total_enrichments)})")
    print(f"  ❌ GAP (sin enriquecer):       {gap:,} ({format_percent(gap, total_enrichments)})")

    return status_dict, total_enrichments


def analyze_products_with_cn_not_in_cima(db: Session):
    """Identifica productos que tienen CN pero NO están en CIMA/Nomenclator."""
    print("\n" + "=" * 80)
    print("2. PRODUCTOS CON CN NO ENCONTRADOS EN CIMA/NOMENCLATOR")
    print("   (Candidatos a catálogo BOT - parafarmacia oficial)")
    print("=" * 80)

    # Buscar ventas con CN que están en manual_review o failed
    # y que tienen codigo_nacional pero no tienen product_catalog_id

    query = db.query(
        SalesData.codigo_nacional,
        SalesData.product_name,
        func.count(SalesData.id).label('veces_vendido'),
        func.sum(SalesData.total_amount).label('total_ventas'),
        func.count(func.distinct(SalesData.pharmacy_id)).label('num_farmacias')
    ).join(
        SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id
    ).filter(
        and_(
            SalesData.codigo_nacional.isnot(None),
            SalesData.codigo_nacional != '',
            SalesEnrichment.enrichment_status.in_(['manual_review', 'failed']),
            SalesEnrichment.product_catalog_id.is_(None)
        )
    ).group_by(
        SalesData.codigo_nacional,
        SalesData.product_name
    ).order_by(
        func.count(SalesData.id).desc()
    ).limit(50)

    results = query.all()

    print(f"\nProductos con CN pero sin match en CIMA/Nomenclator (Top 50):")
    print("-" * 100)
    print(f"{'CN':<12} {'Producto':<45} {'Ventas':>8} {'€ Total':>12} {'Farmacias':>10}")
    print("-" * 100)

    total_ventas_cn = 0
    total_amount_cn = Decimal('0')
    unique_cns = set()

    for row in results:
        cn = row.codigo_nacional or 'N/A'
        name = (row.product_name or 'Sin nombre')[:43]
        ventas = row.veces_vendido or 0
        amount = row.total_ventas or Decimal('0')
        farmacias = row.num_farmacias or 0

        print(f"{cn:<12} {name:<45} {ventas:>8,} {format_currency(amount):>12} {farmacias:>10}")

        total_ventas_cn += ventas
        total_amount_cn += amount
        if cn != 'N/A':
            unique_cns.add(cn)

    print("-" * 100)
    print(f"\n📊 RESUMEN - Productos con CN sin cobertura CIMA:")
    print(f"  CNs únicos encontrados: {len(unique_cns):,}")
    print(f"  Total ventas afectadas: {total_ventas_cn:,}")
    print(f"  Importe total:          {format_currency(total_amount_cn)}")

    return unique_cns, total_ventas_cn, total_amount_cn


def analyze_products_without_cn(db: Session):
    """Identifica productos SIN código nacional (códigos internos puros)."""
    print("\n" + "=" * 80)
    print("3. PRODUCTOS SIN CÓDIGO NACIONAL (Códigos internos)")
    print("   (Requieren catálogo crowdsourced xfarma_free)")
    print("=" * 80)

    # Buscar ventas sin CN que están en manual_review o failed
    query = db.query(
        SalesData.product_name,
        SalesData.ean13,
        func.count(SalesData.id).label('veces_vendido'),
        func.sum(SalesData.total_amount).label('total_ventas'),
        func.count(func.distinct(SalesData.pharmacy_id)).label('num_farmacias')
    ).join(
        SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id
    ).filter(
        and_(
            or_(
                SalesData.codigo_nacional.is_(None),
                SalesData.codigo_nacional == ''
            ),
            SalesEnrichment.enrichment_status.in_(['manual_review', 'failed'])
        )
    ).group_by(
        SalesData.product_name,
        SalesData.ean13
    ).order_by(
        func.count(SalesData.id).desc()
    ).limit(50)

    results = query.all()

    print(f"\nProductos SIN CN (Top 50 por frecuencia):")
    print("-" * 100)
    print(f"{'Producto':<50} {'EAN13':<15} {'Ventas':>8} {'€ Total':>12} {'Farmacias':>10}")
    print("-" * 100)

    total_ventas_sin_cn = 0
    total_amount_sin_cn = Decimal('0')
    unique_products = set()

    for row in results:
        name = (row.product_name or 'Sin nombre')[:48]
        ean = row.ean13 or 'N/A'
        ventas = row.veces_vendido or 0
        amount = row.total_ventas or Decimal('0')
        farmacias = row.num_farmacias or 0

        print(f"{name:<50} {ean:<15} {ventas:>8,} {format_currency(amount):>12} {farmacias:>10}")

        total_ventas_sin_cn += ventas
        total_amount_sin_cn += amount
        if name:
            unique_products.add(name.lower().strip())

    print("-" * 100)
    print(f"\n📊 RESUMEN - Productos sin CN:")
    print(f"  Productos únicos (aprox): {len(unique_products):,}")
    print(f"  Total ventas afectadas:   {total_ventas_sin_cn:,}")
    print(f"  Importe total:            {format_currency(total_amount_sin_cn)}")

    return unique_products, total_ventas_sin_cn, total_amount_sin_cn


def analyze_by_pharmacy(db: Session):
    """Analiza el gap por farmacia."""
    print("\n" + "=" * 80)
    print("4. ANÁLISIS POR FARMACIA")
    print("=" * 80)

    query = db.query(
        Pharmacy.name,
        func.count(SalesEnrichment.id).label('total_ventas'),
        func.sum(
            func.cast(SalesEnrichment.enrichment_status == 'enriched', Integer)
        ).label('enriquecidas'),
        func.sum(
            func.cast(SalesEnrichment.enrichment_status.in_(['manual_review', 'failed']), Integer)
        ).label('sin_enriquecer')
    ).join(
        SalesData, Pharmacy.id == SalesData.pharmacy_id
    ).join(
        SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id
    ).group_by(
        Pharmacy.id, Pharmacy.name
    ).order_by(
        func.count(SalesEnrichment.id).desc()
    )

    results = query.all()

    if not results:
        print("\nNo hay datos de farmacias.")
        return

    print(f"\nDesglose por farmacia:")
    print("-" * 80)
    print(f"{'Farmacia':<30} {'Total':>10} {'Enriquec.':>10} {'Gap':>10} {'% Gap':>8}")
    print("-" * 80)

    for row in results:
        name = (row.name or 'Sin nombre')[:28]
        total = row.total_ventas or 0
        enriched = row.enriquecidas or 0
        gap = row.sin_enriquecer or 0
        pct_gap = format_percent(gap, total)

        print(f"{name:<30} {total:>10,} {enriched:>10,} {gap:>10,} {pct_gap:>8}")

    print("-" * 80)


def simulate_crowdsourced_catalog(db: Session):
    """Simula cómo sería un catálogo crowdsourced con los datos actuales."""
    print("\n" + "=" * 80)
    print("5. SIMULACIÓN: CATÁLOGO CROWDSOURCED xfarma_free")
    print("=" * 80)

    # Productos únicos que podrían entrar al catálogo
    # Agrupados por nombre normalizado

    query = db.query(
        func.lower(func.trim(SalesData.product_name)).label('nombre_normalizado'),
        func.count(func.distinct(SalesData.pharmacy_id)).label('num_farmacias'),
        func.count(SalesData.id).label('total_ventas'),
        func.sum(SalesData.total_amount).label('importe_total'),
        func.array_agg(func.distinct(SalesData.codigo_nacional)).label('codigos_nacionales'),
        func.array_agg(func.distinct(SalesData.ean13)).label('eans')
    ).join(
        SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id
    ).filter(
        and_(
            SalesEnrichment.enrichment_status.in_(['manual_review', 'failed']),
            SalesData.product_name.isnot(None),
            SalesData.product_name != ''
        )
    ).group_by(
        func.lower(func.trim(SalesData.product_name))
    ).having(
        func.count(SalesData.id) >= 2  # Al menos 2 ventas para ser relevante
    ).order_by(
        func.count(func.distinct(SalesData.pharmacy_id)).desc(),
        func.count(SalesData.id).desc()
    ).limit(100)

    results = query.all()

    print(f"\nTop 100 productos candidatos para catálogo crowdsourced:")
    print("(Ordenados por número de farmacias que lo tienen)")
    print("-" * 110)
    print(f"{'Producto':<45} {'Farmacias':>10} {'Ventas':>10} {'€ Total':>12} {'CN':>10} {'EAN':>15}")
    print("-" * 110)

    multi_pharmacy_products = 0
    single_pharmacy_products = 0
    total_catalog_entries = 0

    for row in results:
        name = (row.nombre_normalizado or 'N/A')[:43]
        farmacias = row.num_farmacias or 0
        ventas = row.total_ventas or 0
        importe = row.importe_total or Decimal('0')

        # Filtrar NULLs y vacíos de los arrays
        cns = [cn for cn in (row.codigos_nacionales or []) if cn]
        eans = [ean for ean in (row.eans or []) if ean]

        cn_str = cns[0] if cns else 'N/A'
        ean_str = eans[0] if eans else 'N/A'

        print(f"{name:<45} {farmacias:>10} {ventas:>10,} {format_currency(importe):>12} {cn_str:>10} {ean_str:>15}")

        total_catalog_entries += 1
        if farmacias > 1:
            multi_pharmacy_products += 1
        else:
            single_pharmacy_products += 1

    print("-" * 110)
    print(f"\n📊 SIMULACIÓN CATÁLOGO CROWDSOURCED:")
    print(f"  Entradas potenciales (con ≥2 ventas): {total_catalog_entries:,}")
    print(f"  Productos en múltiples farmacias:     {multi_pharmacy_products:,} (alta confianza)")
    print(f"  Productos en una sola farmacia:       {single_pharmacy_products:,} (baja confianza)")
    print(f"\n  💡 Productos en múltiples farmacias = mayor probabilidad de ser reales")
    print(f"     y podrían verificarse automáticamente con alta confianza.")


def analyze_product_type_distribution(db: Session):
    """Analiza la distribución de tipos de producto."""
    print("\n" + "=" * 80)
    print("6. DISTRIBUCIÓN POR TIPO DE PRODUCTO")
    print("=" * 80)

    query = db.query(
        SalesEnrichment.product_type,
        func.count(SalesEnrichment.id).label('count'),
        func.sum(SalesData.total_amount).label('importe')
    ).join(
        SalesData, SalesEnrichment.sales_data_id == SalesData.id
    ).group_by(
        SalesEnrichment.product_type
    ).order_by(
        func.count(SalesEnrichment.id).desc()
    )

    results = query.all()

    total = sum(r.count for r in results)

    print(f"\nDistribución actual de tipos de producto:")
    print("-" * 60)
    print(f"{'Tipo':<25} {'Registros':>12} {'%':>8} {'Importe':>15}")
    print("-" * 60)

    for row in results:
        tipo = row.product_type or 'NULL/Sin clasificar'
        count = row.count or 0
        importe = row.importe or Decimal('0')
        pct = format_percent(count, total)

        print(f"{tipo:<25} {count:>12,} {pct:>8} {format_currency(importe):>15}")

    print("-" * 60)


def main():
    """Función principal de análisis."""
    print("\n" + "=" * 80)
    print("ANÁLISIS DE GAP: PRODUCTOS DE VENTA LIBRE SIN COBERTURA")
    print("xFarma - Simulación Catálogo Crowdsourced")
    print("=" * 80)

    db = SessionLocal()

    try:
        # 1. Estado general de enriquecimiento
        status_dict, total = analyze_enrichment_status(db)

        # 2. Productos con CN no en CIMA (candidatos BOT)
        unique_cns, ventas_cn, amount_cn = analyze_products_with_cn_not_in_cima(db)

        # 3. Productos sin CN (códigos internos)
        unique_products, ventas_sin_cn, amount_sin_cn = analyze_products_without_cn(db)

        # 4. Análisis por farmacia
        analyze_by_pharmacy(db)

        # 5. Simulación catálogo crowdsourced
        simulate_crowdsourced_catalog(db)

        # 6. Distribución por tipo de producto
        analyze_product_type_distribution(db)

        # Resumen final
        print("\n" + "=" * 80)
        print("RESUMEN EJECUTIVO")
        print("=" * 80)

        gap_total = status_dict.get('manual_review', 0) + status_dict.get('failed', 0)

        print(f"""
📊 ESTADO ACTUAL:
   - Total ventas: {total:,}
   - Enriquecidas: {status_dict.get('enriched', 0):,} ({format_percent(status_dict.get('enriched', 0), total)})
   - GAP total:    {gap_total:,} ({format_percent(gap_total, total)})

📦 DESGLOSE DEL GAP:
   - Con CN (parafarmacia BOT): {len(unique_cns):,} CNs únicos → {ventas_cn:,} ventas → {format_currency(amount_cn)}
   - Sin CN (códigos internos): {len(unique_products):,} productos → {ventas_sin_cn:,} ventas → {format_currency(amount_sin_cn)}

💡 RECOMENDACIONES:
   1. Productos con CN no en CIMA: Crear xfarma_catalog (CN oficial, datos crowdsourced)
   2. Productos sin CN: Crear xfarma_free_catalog (matching por nombre + características)
   3. Priorizar productos vistos en múltiples farmacias (alta confianza)
   4. Considerar verificación manual para productos de alta rotación
""")

    finally:
        db.close()


if __name__ == "__main__":
    main()
