# migrations/versions/optimize_laboratory_mapping_indexes.py
"""
Optimize laboratory mapping indexes - FASE 2.1 Issue #23
Crear indexes optimizados para laboratory mapping queries tras refactoring con LaboratoryQueryBuilder

Revision ID: optimize_laboratory_indexes
Revises: [previous_revision]
Create Date: 2025-09-25

IMPORTANTE: Esta migración optimiza performance de laboratory mapping
después del refactoring FASE 2.1 que eliminó duplicación SQL
"""

import logging

from alembic import op
from sqlalchemy import text

# revision identifiers, used by Alembic
revision = "optimize_laboratory_indexes"
down_revision = None  # Will be updated by Alembic
branch_labels = None
depends_on = None

logger = logging.getLogger(__name__)


def upgrade():
    """
    Crear indexes optimizados para laboratory mapping queries
    FASE 2.1: Indexes basados en análisis de LaboratoryQueryBuilder
    """
    logger.info("[MIGRATION] Iniciando creación de indexes optimizados para laboratory mapping...")

    # Verificar que la tabla existe
    connection = op.get_bind()

    try:
        # Verificar que la tabla product_catalog existe
        result = connection.execute(
            text(
                """
            SELECT EXISTS (
                SELECT FROM information_schema.tables
                WHERE table_name = 'product_catalog'
            )
        """
            )
        )

        if not result.scalar():
            logger.error("[MIGRATION] Tabla product_catalog no existe. Abortando migración.")
            raise Exception("Tabla product_catalog requerida no encontrada")

        # 1. Index individual en nomen_codigo_laboratorio (más usado en queries)
        logger.info("[MIGRATION] Creando index en nomen_codigo_laboratorio...")
        connection.execute(
            text(
                """
            CREATE INDEX CONCURRENTLY IF NOT EXISTS idx_product_catalog_nomen_codigo
            ON product_catalog (nomen_codigo_laboratorio)
            WHERE nomen_codigo_laboratorio IS NOT NULL
        """
            )
        )

        # 2. Index individual en nomen_laboratorio (para búsquedas por nombre)
        logger.info("[MIGRATION] Creando index en nomen_laboratorio...")
        connection.execute(
            text(
                """
            CREATE INDEX CONCURRENTLY IF NOT EXISTS idx_product_catalog_nomen_laboratorio
            ON product_catalog (nomen_laboratorio)
            WHERE nomen_laboratorio IS NOT NULL
        """
            )
        )

        # 3. Index en nomen_estado (para filtrar por ALTA/BAJA)
        logger.info("[MIGRATION] Creando index en nomen_estado...")
        connection.execute(
            text(
                """
            CREATE INDEX CONCURRENTLY IF NOT EXISTS idx_product_catalog_nomen_estado
            ON product_catalog (nomen_estado)
            WHERE nomen_estado IS NOT NULL
        """
            )
        )

        # 4. Index compuesto optimizado para queries complejas (más selectivo primero)
        logger.info("[MIGRATION] Creando index compuesto optimizado...")
        connection.execute(
            text(
                """
            CREATE INDEX CONCURRENTLY IF NOT EXISTS idx_product_catalog_nomen_composite
            ON product_catalog (nomen_codigo_laboratorio, nomen_estado, nomen_laboratorio)
            WHERE nomen_codigo_laboratorio IS NOT NULL
            AND nomen_laboratorio IS NOT NULL
            AND nomen_estado IS NOT NULL
        """
            )
        )

        # 5. Index específico para laboratorios genéricos (LIKE patterns)
        logger.info("[MIGRATION] Creando index para búsquedas de texto en laboratorios...")
        connection.execute(
            text(
                """
            CREATE INDEX CONCURRENTLY IF NOT EXISTS idx_product_catalog_nomen_lab_text
            ON product_catalog USING gin (upper(nomen_laboratorio) gin_trgm_ops)
            WHERE nomen_laboratorio IS NOT NULL
            AND nomen_estado = 'ALTA'
        """
            )
        )

        # 6. Verificar que los indexes se crearon correctamente
        logger.info("[MIGRATION] Verificando indexes creados...")
        index_check = connection.execute(
            text(
                """
            SELECT indexname, tablename, schemaname
            FROM pg_indexes
            WHERE tablename = 'product_catalog'
            AND indexname LIKE 'idx_product_catalog_nomen%'
            ORDER BY indexname
        """
            )
        )

        created_indexes = index_check.fetchall()
        logger.info(f"[MIGRATION] Indexes creados exitosamente: {len(created_indexes)}")
        for idx in created_indexes:
            logger.info(f"[MIGRATION] - {idx.indexname} en {idx.tablename}")

        # 7. Actualizar estadísticas de la tabla para el optimizer
        logger.info("[MIGRATION] Actualizando estadísticas de tabla...")
        connection.execute(text("ANALYZE product_catalog"))

        logger.info("[MIGRATION] Optimización de indexes de laboratory mapping completada exitosamente")

    except Exception as e:
        logger.error(f"[MIGRATION] Error creando indexes de laboratory mapping: {str(e)}")
        raise


def downgrade():
    """
    Eliminar indexes optimizados de laboratory mapping
    ROLLBACK de FASE 2.1
    """
    logger.info("[MIGRATION] Iniciando rollback de indexes de laboratory mapping...")

    connection = op.get_bind()

    try:
        # Eliminar indexes en orden inverso
        indexes_to_drop = [
            "idx_product_catalog_nomen_lab_text",
            "idx_product_catalog_nomen_composite",
            "idx_product_catalog_nomen_estado",
            "idx_product_catalog_nomen_laboratorio",
            "idx_product_catalog_nomen_codigo",
        ]

        for index_name in indexes_to_drop:
            logger.info(f"[MIGRATION] Eliminando index {index_name}...")
            try:
                connection.execute(text(f"DROP INDEX CONCURRENTLY IF EXISTS {index_name}"))
            except Exception as e:
                logger.warning(f"[MIGRATION] No se pudo eliminar {index_name}: {str(e)}")
                continue

        # Actualizar estadísticas después del rollback
        connection.execute(text("ANALYZE product_catalog"))

        logger.info("[MIGRATION] Rollback de indexes de laboratory mapping completado")

    except Exception as e:
        logger.error(f"[MIGRATION] Error en rollback de indexes: {str(e)}")
        raise


def get_migration_summary():
    """
    Resumen de la migración para documentación
    """
    return {
        "migration_id": revision,
        "phase": "FASE 2.1 - Laboratory Query Deduplication",
        "purpose": "Optimizar performance de queries tras eliminar duplicación SQL",
        "indexes_created": [
            "idx_product_catalog_nomen_codigo - Index en código laboratorio",
            "idx_product_catalog_nomen_laboratorio - Index en nombre laboratorio",
            "idx_product_catalog_nomen_estado - Index en estado (ALTA/BAJA)",
            "idx_product_catalog_nomen_composite - Index compuesto optimizado",
            "idx_product_catalog_nomen_lab_text - Index GIN para búsquedas texto",
        ],
        "performance_impact": "Mejora significativa en laboratory mapping queries",
        "backwards_compatible": True,
        "rollback_safe": True,
    }
