# Sistema de Monitoreo Automático CIMA

## Resumen

El Sistema de Monitoreo Automático CIMA es una solución robusta diseñada para detectar automáticamente cuando la sincronización CIMA se detiene o cuelga, y reiniciarla automáticamente para mantener la disponibilidad del sistema.

## Características Principales

### 🚨 Detección Automática de Estancamiento
- **Heartbeat en Redis**: Sistema de latidos que se actualiza durante la sincronización
- **Timeout configurable**: 20 minutos en desarrollo, 15 minutos en Render (más agresivo)
- **Análisis de progreso**: Detecta si no hay progreso real en los items procesados
- **Diagnóstico detallado**: Información completa sobre la causa del estancamiento

### 🔄 Auto-Recovery Inteligente
- **Límites de seguridad**: Máximo 3 intentos por día (2 en Render)
- **Cooldown entre intentos**: 4 horas para evitar loops infinitos
- **Reset diario**: Contadores se resetean automáticamente a las 2 AM
- **Recovery forzado**: Endpoint manual para casos de emergencia

### 📊 Monitoreo en Tiempo Real
- **Heartbeat con métricas**: Progreso, fase actual, memoria, timestamps
- **Health checks ligeros**: Para monitoreo externo sin sobrecarga
- **Estado completo**: Información detallada del monitor y sincronización
- **Logging estructurado**: Eventos de monitoreo con contexto completo

### ⚙️ Configuración Adaptativa
- **Detección de entorno**: Configuración específica para Render vs desarrollo
- **Timeouts adaptativos**: Más agresivos en producción, más relajados en desarrollo
- **Graceful degradation**: Funciona sin Redis, con funcionalidad reducida

## Arquitectura

### Componentes Principales

#### 1. CimaSyncHeartbeat
```python
# Actualiza el progreso cada 60 segundos
heartbeat.update_heartbeat(
    processed_items=1500,
    total_items=67000,
    current_phase="Descargando productos CIMA - Chunk 5/223",
    chunk_number=5,
    chunk_total=223,
    additional_data={
        "memory_info": {"rss_mb": 245.3},
        "status": "PROCESSING"
    }
)
```

#### 2. CimaSyncStallDetector
```python
# Detecta estancamientos automáticamente
is_stalled, diagnosis = detector.is_sync_stalled()

if is_stalled:
    # diagnosis contiene información detallada:
    # - tiempo desde último heartbeat
    # - fase actual
    # - progreso porcentual
    # - timeout configurado
```

#### 3. CimaSyncAutoRecovery
```python
# Gestiona recovery automático con límites
can_recover, limits = recovery.can_attempt_recovery()

if can_recover:
    success, info = recovery.attempt_recovery()
```

#### 4. CimaSyncMonitor
```python
# Monitor principal que coordina todo
monitor.start_monitoring()  # Inicia loop de monitoreo
status = monitor.get_monitor_status()  # Estado completo
```

### Integración con CatalogMaintenanceService

El sistema se integra automáticamente con la sincronización CIMA existente:

```python
def _enrich_from_cima_with_real_progress(self, db: Session, force_update: bool):
    # Heartbeat inicial
    update_cima_heartbeat(
        processed_items=0,
        total_items=67000,
        current_phase="Iniciando sincronización CIMA"
    )

    # Durante procesamiento (en callback)
    def update_system_status(status, progress, message, stats):
        # ... actualizar sistema existente ...

        # NUEVO: Heartbeat automático
        update_cima_heartbeat(
            processed_items=stats.get("total_processed", 0),
            total_items=stats.get("estimated_total", 67000),
            current_phase=f"Descargando productos CIMA {message}",
            additional_data={
                "memory_info": self._get_memory_info(),
                "status": status
            }
        )

    # Cleanup al finalizar
    cima_sync_monitor.heartbeat.clear_heartbeat()
```

## Endpoints de API

### GET /api/v1/system/cima/monitor/status
**Estado completo del monitor**

```json
{
  "status": "success",
  "data": {
    "monitoring_active": true,
    "sync_heartbeat": {
      "timestamp": 1703123456,
      "processed_items": 15000,
      "total_items": 67000,
      "current_phase": "Descargando productos CIMA - Chunk 50/223",
      "progress_percentage": 22.39,
      "last_update": "2024-12-21T10:30:45Z"
    },
    "stall_detection": {
      "is_stalled": false,
      "diagnosis": {
        "reason": "healthy",
        "status": "active",
        "time_since_heartbeat": 45
      }
    },
    "recovery_status": {
      "can_attempt_recovery": true,
      "limits": {
        "daily_attempts": 1,
        "max_attempts": 3
      }
    }
  }
}
```

### POST /api/v1/system/cima/monitor/start
**Iniciar monitoreo automático**

```json
{
  "status": "success",
  "message": "Monitoreo automático CIMA iniciado correctamente",
  "monitoring_active": true,
  "timestamp": "2024-12-21T10:30:45Z"
}
```

### POST /api/v1/system/cima/monitor/stop
**Detener monitoreo automático**

### GET /api/v1/system/cima/monitor/health
**Health check ligero**

```json
{
  "status": "syncing",
  "message": "Sincronizando: 22.39% - Descargando productos CIMA - Chunk 50/223",
  "monitoring_active": true,
  "sync_progress": {
    "percentage": 22.39,
    "processed": 15000,
    "total": 67000,
    "current_phase": "Descargando productos CIMA - Chunk 50/223",
    "last_update": "2024-12-21T10:30:45Z"
  },
  "timestamp": "2024-12-21T10:30:45Z"
}
```

### POST /api/v1/system/cima/monitor/force-recovery
**Recovery forzado manual**

## Configuración

### Variables de Entorno

```bash
# Habilitar monitoreo en desarrollo
ENABLE_CIMA_MONITORING=true

# El monitoreo se inicia automáticamente en:
# - RENDER=true (producción Render)
# - ENVIRONMENT=production
# - ENABLE_CIMA_MONITORING=true (forzado)
```

### Configuración Interna

```python
class CimaSyncMonitorConfig:
    # Timeouts
    STALL_DETECTION_TIMEOUT = 20 * 60  # 20 min desarrollo
    RENDER_STALL_TIMEOUT = 15 * 60     # 15 min Render

    # Límites de recovery
    MAX_RECOVERY_ATTEMPTS = 3    # 3 intentos desarrollo
    RENDER_MAX_ATTEMPTS = 2      # 2 intentos Render
    RECOVERY_COOLDOWN = 4 * 60 * 60  # 4 horas

    # Intervalos
    HEARTBEAT_INTERVAL = 60      # 60s actualizar heartbeat
    MONITOR_CHECK_INTERVAL = 5 * 60  # 5 min verificar estancamiento
```

## Logs y Monitoreo

### Eventos de Logging

```python
# Heartbeat actualizado
logger.debug("monitor.heartbeat.updated",
            processed=15000, total=67000,
            phase="Descargando productos")

# Estancamiento detectado
logger.warning("monitor.stall_detected",
              time_since_heartbeat=1200,
              phase="Chunk 45/223")

# Recovery exitoso
logger.info("monitor.recovery.successful",
           daily_attempts=2)

# Recovery fallido
logger.error("monitor.recovery.failed",
            reason="restart_failed")
```

### Métricas de Redis

```bash
# Keys de Redis utilizadas
cima:sync:heartbeat      # Heartbeat actual
cima:sync:status         # Estado del sync
cima:sync:metrics        # Métricas de performance
cima:sync:recovery       # Estado de recovery (attempts, cooldowns)
```

## Casos de Uso

### 1. Detección Automática de Estancamiento

**Escenario**: La sincronización CIMA se cuelga en el chunk 45 de 223.

**Flujo**:
1. ✅ Heartbeat se actualiza normalmente hasta chunk 45
2. ⏰ 15-20 minutos sin heartbeat
3. 🚨 Monitor detecta estancamiento
4. 🔄 Auto-recovery inicia nueva sincronización
5. ✅ Nueva sincronización continúa desde chunk 1

### 2. Recovery Manual de Emergencia

**Escenario**: Se necesita forzar recovery ignorando límites.

**Comando**:
```bash
curl -X POST http://localhost:8000/api/v1/system/cima/monitor/force-recovery
```

### 3. Monitoreo Externo

**Escenario**: Sistema de alertas externo monitorea health.

**Comando**:
```bash
# Health check cada 5 minutos
curl http://localhost:8000/api/v1/system/cima/monitor/health
```

## Pruebas

### Script de Pruebas Automatizadas

```bash
cd backend
python scripts/test_cima_monitor.py
```

**Pruebas incluidas**:
1. ✅ Lifecycle del monitor (start/stop)
2. ✅ Health checks y estado
3. ✅ Detección de estancamiento
4. ✅ Recovery forzado
5. 🔄 Sincronización con monitoreo (opcional)

### Pruebas Manuales

```bash
# 1. Iniciar monitoreo
curl -X POST http://localhost:8000/api/v1/system/cima/monitor/start

# 2. Verificar estado
curl http://localhost:8000/api/v1/system/cima/monitor/status

# 3. Iniciar sync CIMA (en otra terminal)
curl -X POST http://localhost:8000/api/v1/system/sync \
  -H "Content-Type: application/json" \
  -d '{"target": "cima", "force": false}'

# 4. Monitorear progreso
watch -n 10 'curl -s http://localhost:8000/api/v1/system/cima/monitor/health | jq'
```

## Troubleshooting

### Problemas Comunes

#### Monitor no inicia
```bash
# Verificar Redis
curl http://localhost:8000/api/v1/system/cima/monitor/start

# Error típico: Redis no disponible
# Solución: Verificar docker-compose y Redis connection
```

#### No se detecta estancamiento
```bash
# Verificar configuración de timeout
curl http://localhost:8000/api/v1/system/cima/monitor/status | jq '.data.config'

# Verificar heartbeat activo
curl http://localhost:8000/api/v1/system/cima/monitor/health
```

#### Recovery no funciona
```bash
# Verificar límites
curl http://localhost:8000/api/v1/system/cima/monitor/status | jq '.data.recovery_status'

# Forzar recovery
curl -X POST http://localhost:8000/api/v1/system/cima/monitor/force-recovery
```

### Logs de Diagnóstico

```bash
# Ver logs del backend
docker-compose logs backend | grep -E "monitor\.|cima\."

# Logs específicos de heartbeat
docker-compose logs backend | grep "monitor.heartbeat"

# Logs de recovery
docker-compose logs backend | grep "monitor.recovery"
```

## Rendimiento e Impacto

### Overhead del Sistema

- **CPU**: < 1% adicional durante sincronización
- **Memoria**: ~5MB para estructuras de monitoreo
- **Redis**: ~1KB por heartbeat, ~5KB total por sync
- **Red**: Heartbeat cada 60s (negligible)

### Beneficios

✅ **Disponibilidad**: Auto-recovery reduce downtime de horas a minutos
✅ **Observabilidad**: Visibilidad completa del proceso de sincronización
✅ **Robustez**: Límites previenen loops infinitos y abuso de recursos
✅ **Maintenance**: Reduce intervención manual en producción

## Roadmap Futuro

### v2.0 - Monitoreo Avanzado
- [ ] Predicción de estancamientos basada en patrones
- [ ] Métricas de performance históricas
- [ ] Alertas proactivas por Slack/email
- [ ] Dashboard visual de monitoreo

### v2.1 - Auto-tuning
- [ ] Ajuste dinámico de timeouts según performance
- [ ] Detección de patrones de fallos recurrentes
- [ ] Optimización automática de chunk sizes

---

**Versión**: 1.0
**Última actualización**: 2024-12-21
**Compatibilidad**: xFarma Backend v1.0+
