#!/usr/bin/env python3
"""
Script para crear migración de modelos de enriquecimiento usando Docker
Ejecutar dentro del contenedor backend
"""

import os
import sys


def create_migration():
    """Crea migración usando Alembic dentro del contenedor"""
    print("CREATING ENRICHMENT MIGRATION WITH DOCKER")
    print("=" * 50)

    try:
        # Importar modelos para registrarlos
        from app.database import Base, engine
        from app.models import ProductCatalog, SalesData, SalesEnrichment

        print("Models imported successfully:")
        print(f"- ProductCatalog: {ProductCatalog.__tablename__}")
        print(f"- SalesEnrichment: {SalesEnrichment.__tablename__}")
        print(f"- SalesData: {SalesData.__tablename__} (with enrichment relationship)")

        # Crear todas las tablas nuevas
        print("\nCreating new tables...")
        Base.metadata.create_all(bind=engine)

        print("SUCCESS: Enrichment tables created!")

        # Verificar que las tablas existen
        from sqlalchemy import inspect

        inspector = inspect(engine)
        tables = inspector.get_table_names()

        print(f"\nExisting tables ({len(tables)}):")
        for table in sorted(tables):
            print(f"  - {table}")

        # Verificar tablas específicas de enriquecimiento
        enrichment_tables = ["product_catalog", "sales_enrichment"]
        for table in enrichment_tables:
            if table in tables:
                columns = inspector.get_columns(table)
                print(f"\n{table.upper()} ({len(columns)} columns):")
                for i, col in enumerate(columns[:10]):  # Mostrar primeras 10
                    print(f"  {i+1}. {col['name']}: {col['type']}")
                if len(columns) > 10:
                    print(f"  ... and {len(columns) - 10} more columns")
            else:
                print(f"\nWARNING: {table} not found!")

        return True

    except Exception as e:
        print(f"ERROR creating migration: {e}")
        import traceback

        traceback.print_exc()
        return False


def test_models():
    """Prueba básica de los modelos"""
    print("\nTESTING ENRICHMENT MODELS")
    print("-" * 30)

    try:
        from app.database import SessionLocal
        from app.models import ProductCatalog

        session = SessionLocal()

        # Test ProductCatalog
        print("Testing ProductCatalog...")
        test_product = ProductCatalog(
            commercial_name="TEST PARACETAMOL 500MG",
            national_code="TEST123",
            active_ingredient="Paracetamol",
            atc_code="N02BE01",
            laboratory="TEST LAB",
            is_generic=True,
            official_price=2.50,
        )

        session.add(test_product)
        session.commit()

        # Buscar el producto
        found = session.query(ProductCatalog).filter(ProductCatalog.national_code == "TEST123").first()

        if found:
            print(f"  OK: Product created - {found.commercial_name}")
            print(f"  Price: {found.official_price}€")

            # Limpiar test data
            session.delete(found)
            session.commit()
            print("  Test data cleaned")
        else:
            print("  ERROR: Product not found")

        session.close()
        return True

    except Exception as e:
        print(f"ERROR testing models: {e}")
        return False


def main():
    """Función principal"""
    print("XFARMA ENRICHMENT MODELS - DOCKER MIGRATION")
    print("=" * 60)

    # Verificar que estamos en el contenedor
    if not os.path.exists("/app"):
        print("WARNING: Not running in Docker container")
        print("This script should be run inside the backend container")

    # Crear migración
    migration_ok = create_migration()

    if migration_ok:
        # Probar modelos
        test_ok = test_models()

        if test_ok:
            print("\n" + "=" * 60)
            print("SUCCESS: Enrichment models ready in Docker!")
            print("\nDatabase schema includes:")
            print("- product_catalog: CIMA/nomenclator master catalog")
            print("- sales_enrichment: Enriched sales analysis")
            print("- Updated relationships for enrichment workflow")
            print("\nNext steps:")
            print("1. Create enrichment APIs")
            print("2. Test with real CIMA data")
            print("3. Integrate with frontend dashboard")
            return True

    print("\nERROR: Migration failed")
    return False


if __name__ == "__main__":
    success = main()
    sys.exit(0 if success else 1)
