"""
Utilidades de validación para xFarma
Incluye validaciones específicas para el contexto español
"""

import re
from typing import Optional


def validate_dni_nie(dni_nie: str) -> bool:
    """
    Valida el formato de un DNI o NIE español

    DNI formato: 8 dígitos + 1 letra (ej: 12345678A)
    NIE formato: letra (X,Y,Z) + 7 dígitos + 1 letra (ej: X1234567A)

    Args:
        dni_nie: String con el DNI o NIE a validar

    Returns:
        True si el formato es válido, False en caso contrario
    """
    if not dni_nie:
        return False

    # Limpiar y normalizar
    dni_nie = dni_nie.upper().strip().replace("-", "").replace(" ", "")

    # Patrón para DNI (8 dígitos + letra)
    dni_pattern = r"^[0-9]{8}[A-Z]$"

    # Patrón para NIE (X/Y/Z + 7 dígitos + letra)
    nie_pattern = r"^[XYZ][0-9]{7}[A-Z]$"

    # Validar formato
    if not (re.match(dni_pattern, dni_nie) or re.match(nie_pattern, dni_nie)):
        return False

    # Validar letra de control
    letters = "TRWAGMYFPDXBNJZSQVHLCKE"

    # Para NIE, convertir letra inicial a número
    if dni_nie[0] in "XYZ":
        nie_map = {"X": "0", "Y": "1", "Z": "2"}
        number_str = nie_map[dni_nie[0]] + dni_nie[1:8]
    else:
        number_str = dni_nie[0:8]

    try:
        number = int(number_str)
        expected_letter = letters[number % 23]
        return dni_nie[-1] == expected_letter
    except (ValueError, IndexError):
        return False


def validate_spanish_phone(phone: str) -> bool:
    """
    Valida un número de teléfono español

    Formatos válidos:
    - +34 600123456
    - 600123456
    - 91 234 56 78
    - +34912345678

    Args:
        phone: Número de teléfono a validar

    Returns:
        True si el formato es válido, False en caso contrario
    """
    if not phone:
        return False

    # Limpiar formato
    phone = re.sub(r"[\s\-\(\)]", "", phone)

    # Patrones válidos
    patterns = [
        r"^\+34[6-9][0-9]{8}$",  # Móvil con prefijo internacional
        r"^[6-9][0-9]{8}$",  # Móvil sin prefijo
        r"^\+349[0-9]{8}$",  # Fijo con prefijo internacional
        r"^9[0-9]{8}$",  # Fijo sin prefijo
    ]

    return any(re.match(pattern, phone) for pattern in patterns)


def validate_nif_cif(nif_cif: str) -> bool:
    """
    Valida el formato de un NIF (persona física) o CIF (empresa) español

    NIF: mismo que DNI
    CIF formato: letra + 7 dígitos + dígito/letra control (ej: A12345678)

    Args:
        nif_cif: String con el NIF o CIF a validar

    Returns:
        True si el formato es válido, False en caso contrario
    """
    if not nif_cif:
        return False

    nif_cif = nif_cif.upper().strip().replace("-", "").replace(" ", "")

    # Si empieza con número, es un NIF (validar como DNI)
    if nif_cif[0].isdigit():
        return validate_dni_nie(nif_cif)

    # Validar CIF
    cif_pattern = r"^[ABCDEFGHJKLMNPQRSUVW][0-9]{7}[0-9A-J]$"

    if not re.match(cif_pattern, nif_cif):
        return False

    # Validación adicional del dígito de control para CIF
    # (simplificada, la validación completa es más compleja)
    return True


def validate_pharmacy_license(license_number: str) -> bool:
    """
    Valida el número de colegiado de farmacia

    Formato típico: números y/o letras con guiones
    Ejemplo: COL-2024-0001, 12345, MAD-1234

    Args:
        license_number: Número de colegiado

    Returns:
        True si el formato parece válido
    """
    if not license_number:
        return False

    # Patrón flexible para números de colegiado
    pattern = r"^[A-Z0-9\-\/]{3,20}$"
    return bool(re.match(pattern, license_number.upper()))


def sanitize_dni_nie(dni_nie: Optional[str]) -> Optional[str]:
    """
    Limpia y normaliza un DNI/NIE

    Args:
        dni_nie: DNI/NIE a limpiar

    Returns:
        DNI/NIE normalizado o None si no es válido
    """
    if not dni_nie:
        return None

    # Limpiar y normalizar
    cleaned = dni_nie.upper().strip().replace("-", "").replace(" ", "")

    # Validar
    if validate_dni_nie(cleaned):
        return cleaned

    return None


def format_spanish_phone(phone: str) -> str:
    """
    Formatea un número de teléfono español

    Args:
        phone: Número a formatear

    Returns:
        Número formateado o el original si no es válido
    """
    if not phone:
        return phone

    # Limpiar
    cleaned = re.sub(r"[\s\-\(\)]", "", phone)

    # Si tiene prefijo internacional
    if cleaned.startswith("+34"):
        number = cleaned[3:]
        if len(number) == 9:
            if number[0] in "6789":
                # Móvil: +34 600 123 456
                return f"+34 {number[0:3]} {number[3:6]} {number[6:9]}"
            else:
                # Fijo: +34 91 234 56 78
                return f"+34 {number[0:2]} {number[2:5]} {number[5:7]} {number[7:9]}"

    # Sin prefijo internacional
    elif len(cleaned) == 9:
        if cleaned[0] in "6789":
            # Móvil: 600 123 456
            return f"{cleaned[0:3]} {cleaned[3:6]} {cleaned[6:9]}"
        else:
            # Fijo: 91 234 56 78
            return f"{cleaned[0:2]} {cleaned[2:5]} {cleaned[5:7]} {cleaned[7:9]}"

    return phone
