# backend/app/utils/timezone.py
"""
Utilidades para manejo consistente de timezone Madrid en xFarma
"""
from datetime import datetime, timezone

import pytz

# Zona horaria oficial de Madrid
MADRID_TZ = pytz.timezone("Europe/Madrid")


def now_madrid() -> datetime:
    """
    Obtiene la fecha y hora actual en zona horaria de Madrid.

    Returns:
        datetime: Timestamp actual en Madrid con timezone info
    """
    return datetime.now(MADRID_TZ)


def utc_to_madrid(utc_dt: datetime) -> datetime:
    """
    Convierte datetime UTC a zona horaria de Madrid.

    Args:
        utc_dt: Datetime en UTC

    Returns:
        datetime: Datetime convertido a Madrid
    """
    if utc_dt.tzinfo is None:
        # Asumir que es UTC si no tiene timezone
        utc_dt = utc_dt.replace(tzinfo=timezone.utc)

    return utc_dt.astimezone(MADRID_TZ)


def madrid_to_utc(madrid_dt: datetime) -> datetime:
    """
    Convierte datetime de Madrid a UTC para almacenamiento en BD.

    Args:
        madrid_dt: Datetime en Madrid

    Returns:
        datetime: Datetime convertido a UTC
    """
    if madrid_dt.tzinfo is None:
        # Asumir que es Madrid si no tiene timezone
        madrid_dt = MADRID_TZ.localize(madrid_dt)

    return madrid_dt.astimezone(timezone.utc)


def format_madrid_datetime(dt: datetime, format_str: str = "%d/%m/%Y %H:%M:%S") -> str:
    """
    Formatea datetime en formato español de Madrid.

    Args:
        dt: Datetime a formatear
        format_str: Formato de salida (por defecto español DD/MM/YYYY)

    Returns:
        str: Fecha formateada en Madrid
    """
    if dt.tzinfo is None:
        # Asumir UTC si no tiene timezone
        dt = dt.replace(tzinfo=timezone.utc)

    madrid_dt = dt.astimezone(MADRID_TZ)
    return madrid_dt.strftime(format_str)


def get_madrid_timezone_info() -> dict:
    """
    Información sobre la zona horaria actual de Madrid.

    Returns:
        dict: Info sobre timezone, offset, DST, etc.
    """
    now = now_madrid()

    return {
        "timezone": "Europe/Madrid",
        "current_time": now.isoformat(),
        "utc_offset": str(now.utcoffset()),
        "dst_active": bool(now.dst()),
        "timezone_name": now.tzname(),
        "formatted": format_madrid_datetime(now),
    }


# Constantes útiles
MADRID_TIMEZONE_NAME = "Europe/Madrid"
SPANISH_DATE_FORMAT = "%d/%m/%Y"
SPANISH_DATETIME_FORMAT = "%d/%m/%Y %H:%M:%S"
ISO_DATETIME_FORMAT = "%Y-%m-%dT%H:%M:%S%z"
