"""
Normalización de texto para xFarma - Defensa en profundidad contra corruption de encoding

Estrategia de 3 capas implementada (2025-11-01):
1. Normalización defensiva en puntos de ingesta (nomenclator, CIMA, parsers)
2. Headers UTF-8 explícitos en toda la stack (middleware, gunicorn, nginx)
3. Tests de regresión de encoding

Aplicado en:
- nomenclator_integration.py (líneas ~367-380)
- cima_integration.py (fetch_all_presentations_batch)
- farmatic_parser.py, farmanager_parser.py (parsers ERP)

Problema resuelto:
- Nombres con encoding corrupto en URLs (ESPA%C3%83%C2%91A - doble encoding UTF-8)
- Caracteres españoles (Ñ, tildes) malformados en Render Frankfurt
- Espacios múltiples, caracteres de control
"""

import re
import unicodedata
from typing import Optional

import structlog

logger = structlog.get_logger(__name__)


def normalize_text(
    text: Optional[str],
    max_length: int = 255,
    context: str = "generic",
    preserve_case: bool = False
) -> Optional[str]:
    """
    Función genérica de normalización de texto para prevenir corruption de encoding.

    Esta función implementa la Capa 1 de la estrategia de defensa en profundidad
    contra problemas de encoding UTF-8 en cloud providers (Render, AWS, Azure).

    Transformaciones aplicadas (en orden):
    1. Detección de valores NaN/None/null de pandas/python (Issue #456)
    2. Normalización Unicode NFD → NFC (combinar caracteres especiales)
       - NFD: Ñ = N + ˜ (2 codepoints separados)
       - NFC: Ñ = Ñ (1 codepoint compuesto)
       - Previene doble encoding UTF-8
    3. Eliminar caracteres de control (0x00-0x1F, 0x7F-0x9F)
       - Preservar caracteres imprimibles españoles (Ñ, á, é, í, ó, ú, ü, ¿, ¡)
    4. Normalizar espacios múltiples a uno solo
    5. Eliminar espacios al inicio y final (trim)
    6. Limitar longitud máxima (importante para DB constraints)
    7. Opcional: preservar capitalización original

    Args:
        text: Texto a normalizar (puede ser None o string vacío)
        max_length: Longitud máxima del texto (default: 255 para DB VARCHAR)
        context: Contexto de la normalización para logging ("laboratory", "product", "ingredient", etc.)
        preserve_case: Si True, no modifica capitalización (default: False)

    Returns:
        Texto normalizado o None si el input es inválido

    Examples:
        >>> normalize_text("TECNIMEDE ESPAÑA INDUSTRIA FARMACEUTICA S.A.")
        "TECNIMEDE ESPAÑA INDUSTRIA FARMACEUTICA S.A."

        >>> normalize_text("  NORMON  S.A.  ", context="laboratory")
        "NORMON S.A."

        >>> normalize_text("José María", max_length=10)
        "José María"

        >>> normalize_text("ESPA\u00c3\u0091A")  # Doble encoding simulado
        "ESPAÑA"

        >>> normalize_text(None)
        None

        >>> normalize_text("nan")  # Issue #456: pandas NaN convertido a string
        None
    """
    if not text or not isinstance(text, str):
        return None

    # Issue #456: Detectar valores NaN/None/null de pandas/python
    # Cuando pandas lee CSV con valores vacíos, los convierte a NaN (float)
    # str(NaN) produce el string literal 'nan' que se guarda en DB
    # Esto causa que los productos aparezcan como "Sin identificar"
    INVALID_STRINGS = {"nan", "none", "null", "n/a", "na", "<na>", "undefined"}
    if text.strip().lower() in INVALID_STRINGS:
        return None

    # 1. Normalización Unicode: NFD → NFC
    # CRÍTICO: Previene doble encoding UTF-8 (root cause del problema Render)
    # NFD (Canonical Decomposition): Descompone caracteres (Ñ → N + ˜)
    # NFC (Canonical Composition): Compone caracteres (N + ˜ → Ñ)
    normalized = unicodedata.normalize('NFC', text)

    # 2. Eliminar caracteres de control y no imprimibles
    # Categoría 'C' de Unicode incluye: Cc (control), Cf (format), Cs (surrogate)
    # Preservar: Ñ (\u00D1), tildes (á \u00E1, é \u00E9, etc.), ¿ (\u00BF), ¡ (\u00A1)
    normalized = ''.join(
        char for char in normalized
        if not unicodedata.category(char).startswith('C')
    )

    # 3. Normalizar espacios múltiples a uno solo
    # "NORMON  S.A." → "NORMON S.A."
    normalized = re.sub(r'\s+', ' ', normalized)

    # 4. Eliminar espacios al inicio y final
    normalized = normalized.strip()

    # 5. Limitar longitud máxima (importante para DB constraints)
    if len(normalized) > max_length:
        logger.warning(
            f"text.{context}.truncated",
            original_length=len(normalized),
            max_length=max_length,
            text_preview=normalized[:50] + "...",
            context=context
        )
        normalized = normalized[:max_length]

    # 6. Validar que el resultado final no esté vacío
    if not normalized:
        logger.warning(
            f"text.{context}.empty_after_normalization",
            original_text=text[:100] if text else None,
            context=context
        )
        return None

    return normalized


def normalize_laboratory_name(name: Optional[str], max_length: int = 255) -> Optional[str]:
    """
    Normaliza el nombre de un laboratorio farmacéutico.

    Wrapper de normalize_text() con contexto "laboratory".
    Mantiene la API existente para compatibilidad con código legacy.

    Args:
        name: Nombre del laboratorio (puede ser None o string vacío)
        max_length: Longitud máxima del nombre (default: 255 para DB)

    Returns:
        Nombre normalizado o None si el input es inválido

    Examples:
        >>> normalize_laboratory_name("TECNIMEDE ESPAÑA INDUSTRIA FARMACEUTICA S.A.")
        "TECNIMEDE ESPAÑA INDUSTRIA FARMACEUTICA S.A."

        >>> normalize_laboratory_name("  NORMON  S.A.  ")
        "NORMON S.A."

        >>> normalize_laboratory_name("KERN  PHARMA,  S.L.")
        "KERN PHARMA, S.L."

        >>> normalize_laboratory_name(None)
        None
    """
    return normalize_text(text=name, max_length=max_length, context="laboratory")


def normalize_laboratory_code(code: Optional[str]) -> Optional[str]:
    """
    Normaliza el código de un laboratorio (código del nomenclátor).

    Transformaciones:
    - Eliminar espacios
    - Eliminar ".0" de códigos decimales (ej: "1237.0" → "1237")
    - Convertir a uppercase
    - Limitar longitud a 20 caracteres

    Args:
        code: Código del laboratorio (puede ser None o string vacío)

    Returns:
        Código normalizado o None si el input es inválido

    Examples:
        >>> normalize_laboratory_code("1237.0")
        "1237"

        >>> normalize_laboratory_code("  abc123  ")
        "ABC123"

        >>> normalize_laboratory_code(None)
        None
    """
    if not code or not isinstance(code, str):
        return None

    # Eliminar espacios
    normalized = code.strip()

    # Eliminar ".0" de códigos decimales
    if normalized.endswith(".0"):
        normalized = normalized[:-2]

    # Convertir a uppercase para consistencia
    normalized = normalized.upper()

    # Limitar longitud
    if len(normalized) > 20:
        logger.warning(
            "laboratory.code.truncated",
            original_length=len(normalized),
            code_preview=normalized[:20]
        )
        normalized = normalized[:20]

    # Validar que no esté vacío
    if not normalized or normalized == "NAN":
        return None

    return normalized


def validate_laboratory_name_for_api(name: str) -> bool:
    """
    Valida que un nombre de laboratorio sea apto para usar en APIs.

    Criterios:
    - No contiene caracteres de control
    - Longitud entre 3 y 255 caracteres
    - No contiene caracteres peligrosos para URL encoding
    - Es una string válida UTF-8

    Args:
        name: Nombre del laboratorio a validar

    Returns:
        True si el nombre es válido para APIs

    Examples:
        >>> validate_laboratory_name_for_api("NORMON S.A.")
        True

        >>> validate_laboratory_name_for_api("AB")  # Muy corto
        False

        >>> validate_laboratory_name_for_api("NOMBRE\x00CON\x00NULLS")  # Caracteres de control
        False
    """
    if not name or not isinstance(name, str):
        return False

    # Validar longitud
    if len(name) < 3 or len(name) > 255:
        return False

    # Validar que no contiene caracteres de control
    if any(unicodedata.category(char).startswith('C') for char in name):
        return False

    # Validar que es UTF-8 válido
    try:
        name.encode('utf-8')
    except UnicodeEncodeError:
        return False

    return True
