"""
Subscription Helper Functions

Utilidades para gestionar restricciones y límites de planes de suscripción.
"""

import logging
from datetime import datetime, timedelta, timezone
from typing import Optional
from sqlalchemy import func
from sqlalchemy.orm import Session

from ..models.user import User
from ..models.sales_data import SalesData

logger = logging.getLogger(__name__)


def get_data_date_limit_for_user(user: User, db: Session) -> Optional[datetime]:
    """
    Calcula la fecha límite de datos para usuarios según su plan de suscripción.

    Para usuarios FREE:
    - Solo pueden acceder a datos de los últimos 3 meses
    - Los 3 meses se calculan desde la fecha de su última venta registrada
    - Si no hay ventas, retorna None (sin restricción temporal)

    Para usuarios PRO y MAX:
    - Sin restricciones temporales (retorna None)

    Args:
        user: Usuario actual
        db: Sesión de base de datos

    Returns:
        datetime: Fecha límite (datos >= esta fecha son accesibles)
        None: Sin restricción temporal

    Example:
        >>> date_limit = get_data_date_limit_for_user(current_user, db)
        >>> if date_limit:
        ...     query = query.filter(SalesData.sale_date >= date_limit)
    """
    # Solo usuarios FREE tienen restricción temporal
    if user.subscription_plan != "free":
        return None

    # Verificar que el usuario tenga farmacia asignada
    if not user.pharmacy_id:
        return None

    # Obtener fecha de última venta del usuario
    last_sale_date = db.query(func.max(SalesData.sale_date)).filter(
        SalesData.pharmacy_id == user.pharmacy_id
    ).scalar()

    # Si no hay ventas, no hay restricción
    if not last_sale_date:
        return None

    # Calcular límite: 3 meses (90 días) hacia atrás desde última venta
    # Fix: Asegurar que datetime sea aware (UTC) para evitar comparación naive vs aware
    if last_sale_date.tzinfo is None:
        last_sale_date = last_sale_date.replace(tzinfo=timezone.utc)
    date_limit = last_sale_date - timedelta(days=90)

    # Log para audit trail y debugging
    logger.info(
        f"FREE tier restriction calculated: user={user.email}, "
        f"last_sale={last_sale_date.strftime('%Y-%m-%d')}, "
        f"limit={date_limit.strftime('%Y-%m-%d')}"
    )

    return date_limit


def has_data_access(user: User, sale_date: datetime, db: Session) -> bool:
    """
    Verifica si un usuario tiene acceso a datos de una fecha específica.

    Args:
        user: Usuario a verificar
        sale_date: Fecha de la venta a verificar
        db: Sesión de base de datos

    Returns:
        bool: True si el usuario puede acceder a datos de esa fecha

    Example:
        >>> if has_data_access(current_user, sale.sale_date, db):
        ...     # Mostrar datos
    """
    date_limit = get_data_date_limit_for_user(user, db)

    # Sin restricción
    if date_limit is None:
        return True

    # Verificar si la fecha está dentro del rango permitido
    return sale_date >= date_limit


def get_accessible_months_count(user: User) -> Optional[int]:
    """
    Retorna el número de meses de datos accesibles según el plan.

    Args:
        user: Usuario actual

    Returns:
        int: Número de meses accesibles
        None: Sin restricción (acceso ilimitado)
    """
    plan_limits = {
        "free": 3,
        "pro": None,  # Sin restricción
        "max": None   # Sin restricción
    }

    return plan_limits.get(user.subscription_plan or "free", 3)
