# backend/app/utils/spanish_holidays.py
"""
Utilidad para festivos nacionales españoles.

Issue #501: Fase 4 - Detector de Anomalías + Alertas.

Proporciona lista de festivos para filtrar falsos positivos en detección de anomalías.
Los festivos suelen tener patrones de venta atípicos que no son verdaderas anomalías.
"""

from datetime import date
from typing import List, Set


# Festivos nacionales fijos (se repiten cada año)
FIXED_NATIONAL_HOLIDAYS = [
    (1, 1),   # Año Nuevo
    (1, 6),   # Epifanía del Señor (Reyes)
    (5, 1),   # Fiesta del Trabajo
    (8, 15),  # Asunción de la Virgen
    (10, 12), # Fiesta Nacional de España
    (11, 1),  # Todos los Santos
    (12, 6),  # Día de la Constitución
    (12, 8),  # Inmaculada Concepción
    (12, 25), # Navidad
]

# Festivos móviles (Semana Santa) - calculados dinámicamente
# Jueves Santo y Viernes Santo dependen de la Pascua


def _calculate_easter(year: int) -> date:
    """
    Calcula la fecha de Pascua usando el algoritmo de Butcher.

    Args:
        year: Año para calcular

    Returns:
        Fecha del Domingo de Pascua
    """
    a = year % 19
    b = year // 100
    c = year % 100
    d = b // 4
    e = b % 4
    f = (b + 8) // 25
    g = (b - f + 1) // 3
    h = (19 * a + b - d - g + 15) % 30
    i = c // 4
    k = c % 4
    l = (32 + 2 * e + 2 * i - h - k) % 7
    m = (a + 11 * h + 22 * l) // 451
    month = (h + l - 7 * m + 114) // 31
    day = ((h + l - 7 * m + 114) % 31) + 1
    return date(year, month, day)


def get_easter_holidays(year: int) -> List[date]:
    """
    Obtiene los festivos de Semana Santa para un año.

    Args:
        year: Año

    Returns:
        Lista con Jueves Santo y Viernes Santo
    """
    easter = _calculate_easter(year)

    # Jueves Santo: 3 días antes de Pascua
    jueves_santo = date.fromordinal(easter.toordinal() - 3)

    # Viernes Santo: 2 días antes de Pascua
    viernes_santo = date.fromordinal(easter.toordinal() - 2)

    return [jueves_santo, viernes_santo]


def get_spanish_holidays(year: int) -> Set[date]:
    """
    Obtiene todos los festivos nacionales españoles para un año.

    Args:
        year: Año

    Returns:
        Set de fechas de festivos
    """
    holidays = set()

    # Festivos fijos
    for month, day in FIXED_NATIONAL_HOLIDAYS:
        holidays.add(date(year, month, day))

    # Festivos móviles (Semana Santa)
    for holiday in get_easter_holidays(year):
        holidays.add(holiday)

    return holidays


def get_spanish_holidays_range(start_date: date, end_date: date) -> Set[date]:
    """
    Obtiene festivos españoles en un rango de fechas.

    Args:
        start_date: Fecha inicio
        end_date: Fecha fin

    Returns:
        Set de fechas de festivos en el rango
    """
    holidays = set()

    # Obtener años en el rango
    for year in range(start_date.year, end_date.year + 1):
        year_holidays = get_spanish_holidays(year)
        for holiday in year_holidays:
            if start_date <= holiday <= end_date:
                holidays.add(holiday)

    return holidays


def is_spanish_holiday(check_date: date) -> bool:
    """
    Verifica si una fecha es festivo nacional español.

    Args:
        check_date: Fecha a verificar

    Returns:
        True si es festivo
    """
    return check_date in get_spanish_holidays(check_date.year)


def get_holiday_name(check_date: date) -> str | None:
    """
    Obtiene el nombre del festivo para una fecha.

    Args:
        check_date: Fecha a verificar

    Returns:
        Nombre del festivo o None si no es festivo
    """
    # Festivos fijos
    fixed_names = {
        (1, 1): "Año Nuevo",
        (1, 6): "Epifanía del Señor (Reyes)",
        (5, 1): "Fiesta del Trabajo",
        (8, 15): "Asunción de la Virgen",
        (10, 12): "Fiesta Nacional de España",
        (11, 1): "Todos los Santos",
        (12, 6): "Día de la Constitución",
        (12, 8): "Inmaculada Concepción",
        (12, 25): "Navidad",
    }

    key = (check_date.month, check_date.day)
    if key in fixed_names:
        return fixed_names[key]

    # Comprobar Semana Santa (siempre devuelve 2 fechas)
    easter_holidays = get_easter_holidays(check_date.year)
    if check_date == easter_holidays[0]:
        return "Jueves Santo"
    if check_date == easter_holidays[1]:
        return "Viernes Santo"

    return None
