﻿"""
Utilidades de seguridad y guardas para proteger operaciones críticas en producción.
Implementa controles de ambiente, logging de auditoría y validaciones de seguridad.
"""

import logging
import os
from functools import wraps
from typing import Any, Callable, Dict, Literal, Optional

import structlog
from sqlalchemy.orm import Session

from app.utils.datetime_utils import utc_now

from ..models.system_health import DeveloperLog

logger = logging.getLogger(__name__)
struct_logger = structlog.get_logger(__name__)

# Tipos de ambiente
EnvironmentType = Literal["development", "production", "test", "staging"]


class ProductionSafetyGuard:
    """
    Guardas de seguridad para operaciones críticas en producción.

    Funcionalidades:
    - Detección automática de ambiente
    - Validaciones de seguridad antes de operaciones destructivas
    - Logging de auditoría obligatorio
    - Bloqueo de operaciones peligrosas en producción sin confirmación explícita
    """

    # Operaciones que requieren confirmación especial en producción
    CRITICAL_OPERATIONS = {
        "delete_all_sales_data",
        "clean_catalog_full",
        "reset_system_configuration",
        "vacuum_database",
        "rebuild_indexes",
        "force_catalog_rebuild",
        "reset_user_sessions",
        "clear_all_cache",
    }

    # Textos de confirmación requeridos por operación
    CONFIRMATION_TEXTS = {
        "delete_all_sales_data": "ELIMINAR VENTAS",
        "clean_catalog_full": "LIMPIAR CATALOGO",
        "reset_system_configuration": "RESET CONFIG",
        "vacuum_database": "VACUUM DB",
        "rebuild_indexes": "REBUILD INDEXES",
        "force_catalog_rebuild": "REBUILD CATALOG",
        "reset_user_sessions": "RESET SESSIONS",
        "clear_all_cache": "CLEAR ALL CACHE",
    }

    @classmethod
    def detect_environment(cls) -> EnvironmentType:
        """
        Detecta el ambiente de ejecución actual con múltiples verificaciones.

        Returns:
            EnvironmentType: Ambiente detectado
        """
        # Verificar variable ENVIRONMENT explícita
        env_var = os.getenv("ENVIRONMENT", "").lower()
        if env_var in ["production", "prod"]:
            return "production"
        elif env_var in ["development", "dev"]:
            return "development"
        elif env_var in ["test", "testing"]:
            return "test"
        elif env_var in ["staging", "stage"]:
            return "staging"

        # Detectar Render production
        if cls.is_render_production():
            return "production"

        # Detectar por URL de base de datos (contiene indicadores de producción)
        db_url = os.getenv("DATABASE_URL", "")
        if any(indicator in db_url.lower() for indicator in ["prod", "render", "amazonaws", "heroku"]):
            return "production"

        # Por defecto, desarrollo
        return "development"

    @classmethod
    def is_render_production(cls) -> bool:
        """
        Detecta específicamente si estamos en Render production.

        Returns:
            bool: True si es Render production
        """
        return (
            os.getenv("RENDER") == "true"
            or "onrender.com" in os.getenv("RENDER_EXTERNAL_URL", "")
            or os.getenv("RENDER_SERVICE_ID") is not None
        )

    @classmethod
    def is_production_environment(cls) -> bool:
        """
        Verifica si el ambiente actual es producción.

        Returns:
            bool: True si es producción
        """
        return cls.detect_environment() == "production"

    @classmethod
    def is_development_environment(cls) -> bool:
        """
        Verifica si el ambiente actual es desarrollo.

        Returns:
            bool: True si es desarrollo
        """
        return cls.detect_environment() == "development"

    @classmethod
    def validate_operation_safety(
        cls, operation_id: str, confirmation_text: Optional[str] = None, admin_override: bool = False
    ) -> tuple[bool, str]:
        """
        Valida si una operación es segura de ejecutar en el ambiente actual.

        Args:
            operation_id: ID de la operación a validar
            confirmation_text: Texto de confirmación proporcionado
            admin_override: Flag de override de administrador

        Returns:
            tuple[bool, str]: (es_segura, mensaje_error)
        """
        environment = cls.detect_environment()

        # En desarrollo, permitir con warnings
        if environment == "development":
            if operation_id in cls.CRITICAL_OPERATIONS:
                struct_logger.warning(
                    "Dangerous operation in development", operation_id=operation_id, environment=environment
                )
            return True, "Operación permitida en desarrollo"

        # En producción, validaciones estrictas
        if environment == "production":
            if operation_id not in cls.CRITICAL_OPERATIONS:
                # Operación no crítica, permitir
                return True, "Operación no crítica permitida"

            # Operación crítica - verificar confirmación
            required_confirmation = cls.CONFIRMATION_TEXTS.get(operation_id)
            if not required_confirmation:
                return False, f"Operación '{operation_id}' no tiene texto de confirmación definido"

            if not confirmation_text:
                return False, f"Texto de confirmación requerido en producción: '{required_confirmation}'"

            if confirmation_text.strip().upper() != required_confirmation.upper():
                return False, f"Confirmación incorrecta. Se requiere: '{required_confirmation}'"

            # Verificar override de administrador para operaciones extremas
            extreme_operations = {"delete_all_sales_data", "clean_catalog_full", "force_catalog_rebuild"}
            if operation_id in extreme_operations and not admin_override:
                return False, f"Operación '{operation_id}' requiere override explícito de administrador en producción"

            return True, "Validación de seguridad superada"

        # Otros ambientes (test, staging)
        return True, f"Operación permitida en ambiente {environment}"


class AuditLogger:
    """
    Sistema de logging de auditoría para operaciones críticas.
    Registra todas las acciones administrativas con contexto completo.
    """

    @classmethod
    def log_operation_attempt(
        cls,
        db: Session,
        operation_id: str,
        user_id: Optional[str] = None,
        environment: Optional[str] = None,
        confirmation_provided: Optional[str] = None,
        success: bool = False,
        error_message: Optional[str] = None,
        execution_time_ms: Optional[float] = None,
        additional_context: Optional[Dict[str, Any]] = None,
    ) -> None:
        """
        Registra intento de operación crítica en logs de auditoría.

        Args:
            db: Sesión de base de datos
            operation_id: ID de la operación
            user_id: ID del usuario que ejecuta
            environment: Ambiente donde se ejecuta
            confirmation_provided: Texto de confirmación proporcionado
            success: Si la operación fue exitosa
            error_message: Mensaje de error si falló
            execution_time_ms: Tiempo de ejecución en milisegundos
            additional_context: Contexto adicional
        """
        try:
            # Detectar ambiente si no se proporciona
            if not environment:
                environment = ProductionSafetyGuard.detect_environment()

            # Construir contexto de auditoría
            audit_context = {
                "operation_id": operation_id,
                "user_id": user_id or "unknown",
                "environment": environment,
                "confirmation_provided": bool(confirmation_provided),
                "success": success,
                "timestamp": utc_now().isoformat(),
                "render_production": ProductionSafetyGuard.is_render_production(),
            }

            # Añadir contexto adicional
            if additional_context:
                audit_context.update(additional_context)

            if error_message:
                audit_context["error_message"] = error_message

            if execution_time_ms:
                audit_context["execution_time_ms"] = execution_time_ms

            # Determinar nivel de log
            log_level = "ERROR" if not success else "WARNING" if environment == "production" else "INFO"

            # Crear mensaje de auditoría
            status = "SUCCESS" if success else "FAILED"
            message = f"[AUDIT] Operación crítica {operation_id} - {status}"

            # Registrar en logs estructurados
            if success:
                struct_logger.warning(message, **audit_context)
            else:
                struct_logger.error(message, **audit_context)

            # Registrar en developer logs si la sesión de BD está disponible
            if db:
                try:
                    developer_log = DeveloperLog(
                        level=log_level,
                        logger_name="audit_logger",
                        message=message,
                        context=audit_context,
                        execution_time_ms=execution_time_ms,
                    )
                    db.add(developer_log)
                    db.commit()
                except Exception as e:
                    # No fallar la operación por error de logging
                    logger.error(f"Error guardando audit log en BD: {e}")
                    db.rollback()

        except Exception as e:
            # Logging de auditoría no debe fallar nunca
            logger.error(f"Error crítico en audit logging: {e}")


def require_safety_validation(operation_id: str):
    """
    Decorador que requiere validación de seguridad antes de ejecutar una función.

    Args:
        operation_id: ID de la operación para validaciones
    """

    def decorator(func: Callable) -> Callable:
        @wraps(func)
        def wrapper(*args, **kwargs):
            # Extraer parámetros comunes
            confirmation_text = kwargs.get("confirmation_text")
            admin_override = kwargs.get("admin_override", False)

            # Validar seguridad de la operación
            is_safe, error_message = ProductionSafetyGuard.validate_operation_safety(
                operation_id=operation_id, confirmation_text=confirmation_text, admin_override=admin_override
            )

            if not is_safe:
                # Log del intento fallido
                from fastapi import HTTPException

                # Intentar obtener sesión de BD de los argumentos
                db = None
                for arg in args:
                    if hasattr(arg, "query"):  # Es una sesión de SQLAlchemy
                        db = arg
                        break

                if db:
                    AuditLogger.log_operation_attempt(
                        db=db,
                        operation_id=operation_id,
                        confirmation_provided=confirmation_text,
                        success=False,
                        error_message=error_message,
                    )

                raise HTTPException(
                    status_code=403, detail=f"Operación bloqueada por validación de seguridad: {error_message}"
                )

            # Ejecutar función con logging de auditoría
            start_time = utc_now()
            success = False
            error_msg = None

            try:
                result = func(*args, **kwargs)
                success = True
                return result
            except Exception as e:
                error_msg = str(e)
                raise
            finally:
                # Log de auditoría post-ejecución
                execution_time = (utc_now() - start_time).total_seconds() * 1000

                # Intentar obtener sesión de BD
                db = None
                for arg in args:
                    if hasattr(arg, "query"):
                        db = arg
                        break

                if db:
                    AuditLogger.log_operation_attempt(
                        db=db,
                        operation_id=operation_id,
                        confirmation_provided=confirmation_text,
                        success=success,
                        error_message=error_msg,
                        execution_time_ms=execution_time,
                    )

        return wrapper

    return decorator


def get_environment_info() -> Dict[str, Any]:
    """
    Obtiene información completa del ambiente actual.

    Returns:
        Dict con información del ambiente
    """
    return {
        "environment": ProductionSafetyGuard.detect_environment(),
        "is_production": ProductionSafetyGuard.is_production_environment(),
        "is_development": ProductionSafetyGuard.is_development_environment(),
        "is_render": ProductionSafetyGuard.is_render_production(),
        "environment_variables": {
            "ENVIRONMENT": os.getenv("ENVIRONMENT"),
            "RENDER": os.getenv("RENDER"),
            "RENDER_SERVICE_ID": os.getenv("RENDER_SERVICE_ID"),
            "DATABASE_URL_MASKED": "***" if os.getenv("DATABASE_URL") else None,
        },
        "safety_enabled": True,
        "audit_logging": True,
    }


# Funciones de conveniencia
def is_production() -> bool:
    """Shortcut para verificar si es producción"""
    return ProductionSafetyGuard.is_production_environment()


def is_development() -> bool:
    """Shortcut para verificar si es desarrollo"""
    return ProductionSafetyGuard.is_development_environment()


def is_render() -> bool:
    """Shortcut para verificar si es Render"""
    return ProductionSafetyGuard.is_render_production()
