# backend/app/utils/retry_helper.py
"""
Helper para implementar retry logic con exponential backoff.
Especialmente útil para APIs externas y rate limiting.
"""

import asyncio
import time
from functools import wraps
from typing import Any, Callable

import structlog

from app.exceptions import APIError, RateLimitError

logger = structlog.get_logger(__name__)


def exponential_backoff(
    max_retries: int = 3,
    initial_delay: float = 1.0,
    max_delay: float = 60.0,
    exponential_base: float = 2.0,
    exceptions: tuple = (RateLimitError, APIError),
    retry_on_rate_limit: bool = True,
) -> Callable:
    """
    Decorador para reintentar operaciones con exponential backoff.

    Args:
        max_retries: Número máximo de reintentos
        initial_delay: Retraso inicial en segundos
        max_delay: Retraso máximo en segundos
        exponential_base: Base para el cálculo exponencial
        exceptions: Tupla de excepciones que disparan reintentos
        retry_on_rate_limit: Si reintentar automáticamente en RateLimitError

    Returns:
        Función decorada con retry logic

    Ejemplo:
        @exponential_backoff(max_retries=3)
        def call_external_api():
            # Código que puede fallar
            pass
    """

    def decorator(func: Callable) -> Callable:
        @wraps(func)
        def sync_wrapper(*args, **kwargs) -> Any:
            """Wrapper para funciones síncronas."""
            attempt = 0
            delay = initial_delay

            while attempt <= max_retries:
                try:
                    # Intentar ejecutar la función
                    return func(*args, **kwargs)

                except exceptions as e:
                    attempt += 1

                    # Si es RateLimitError, usar el tiempo de retry sugerido
                    if isinstance(e, RateLimitError) and retry_on_rate_limit:
                        if hasattr(e, "details") and e.details.get("retry_after"):
                            delay = min(e.details["retry_after"], max_delay)
                            logger.warning(
                                "rate_limit.retry",
                                function=func.__name__,
                                attempt=attempt,
                                retry_after=delay,
                                endpoint=e.details.get("endpoint"),
                            )
                        else:
                            # Usar exponential backoff normal
                            delay = min(delay * exponential_base, max_delay)
                    else:
                        # Para otras excepciones, usar exponential backoff
                        delay = min(delay * exponential_base, max_delay)

                    if attempt > max_retries:
                        logger.error(
                            "retry.max_attempts_exceeded", function=func.__name__, max_retries=max_retries, error=str(e)
                        )
                        raise  # Re-lanzar la última excepción

                    logger.info(
                        "retry.attempting",
                        function=func.__name__,
                        attempt=attempt,
                        delay=delay,
                        error_type=type(e).__name__,
                    )

                    # Esperar antes del siguiente intento
                    time.sleep(delay)

                except Exception as e:
                    # Para excepciones no manejadas, re-lanzar inmediatamente
                    logger.error(
                        "retry.unexpected_error", function=func.__name__, error=str(e), error_type=type(e).__name__
                    )
                    raise

            # No debería llegar aquí, pero por seguridad
            raise Exception(f"Retry logic error in {func.__name__}")

        @wraps(func)
        async def async_wrapper(*args, **kwargs) -> Any:
            """Wrapper para funciones asíncronas."""
            attempt = 0
            delay = initial_delay

            while attempt <= max_retries:
                try:
                    # Intentar ejecutar la función
                    return await func(*args, **kwargs)

                except exceptions as e:
                    attempt += 1

                    # Si es RateLimitError, usar el tiempo de retry sugerido
                    if isinstance(e, RateLimitError) and retry_on_rate_limit:
                        if hasattr(e, "details") and e.details.get("retry_after"):
                            delay = min(e.details["retry_after"], max_delay)
                            logger.warning(
                                "rate_limit.retry_async",
                                function=func.__name__,
                                attempt=attempt,
                                retry_after=delay,
                                endpoint=e.details.get("endpoint"),
                            )
                        else:
                            # Usar exponential backoff normal
                            delay = min(delay * exponential_base, max_delay)
                    else:
                        # Para otras excepciones, usar exponential backoff
                        delay = min(delay * exponential_base, max_delay)

                    if attempt > max_retries:
                        logger.error(
                            "retry.max_attempts_exceeded_async",
                            function=func.__name__,
                            max_retries=max_retries,
                            error=str(e),
                        )
                        raise  # Re-lanzar la última excepción

                    logger.info(
                        "retry.attempting_async",
                        function=func.__name__,
                        attempt=attempt,
                        delay=delay,
                        error_type=type(e).__name__,
                    )

                    # Esperar antes del siguiente intento
                    await asyncio.sleep(delay)

                except Exception as e:
                    # Para excepciones no manejadas, re-lanzar inmediatamente
                    logger.error(
                        "retry.unexpected_error_async",
                        function=func.__name__,
                        error=str(e),
                        error_type=type(e).__name__,
                    )
                    raise

            # No debería llegar aquí, pero por seguridad
            raise Exception(f"Async retry logic error in {func.__name__}")

        # Retornar el wrapper apropiado basado en si la función es async o no
        if asyncio.iscoroutinefunction(func):
            return async_wrapper
        else:
            return sync_wrapper

    return decorator


class RetryHelper:
    """
    Clase helper para operaciones de retry más complejas.
    """

    @staticmethod
    def calculate_delay(
        attempt: int,
        initial_delay: float = 1.0,
        max_delay: float = 60.0,
        exponential_base: float = 2.0,
        jitter: bool = True,
    ) -> float:
        """
        Calcula el delay para un intento dado con optional jitter.

        Args:
            attempt: Número de intento (empezando en 1)
            initial_delay: Delay inicial
            max_delay: Delay máximo
            exponential_base: Base exponencial
            jitter: Si agregar jitter aleatorio para evitar thundering herd

        Returns:
            Delay en segundos
        """
        # Calcular delay exponencial
        delay = min(initial_delay * (exponential_base ** (attempt - 1)), max_delay)

        # Agregar jitter si está habilitado (reduce entre 0-25% el delay)
        if jitter:
            import random

            jitter_amount = delay * random.uniform(0, 0.25)
            delay = delay - jitter_amount

        return delay

    @staticmethod
    async def retry_with_context(
        func: Callable,
        context: dict,
        max_retries: int = 3,
        initial_delay: float = 1.0,
        max_delay: float = 60.0,
        exceptions: tuple = (RateLimitError, APIError),
    ) -> Any:
        """
        Ejecuta una función con retry y contexto adicional para logging.

        Args:
            func: Función a ejecutar
            context: Diccionario con contexto para logging
            max_retries: Número máximo de reintentos
            initial_delay: Delay inicial
            max_delay: Delay máximo
            exceptions: Excepciones que disparan retry

        Returns:
            Resultado de la función

        Raises:
            La última excepción si se agotan los reintentos
        """
        attempt = 0

        while attempt <= max_retries:
            try:
                # Ejecutar función (async o sync)
                if asyncio.iscoroutinefunction(func):
                    return await func()
                else:
                    return func()

            except exceptions as e:
                attempt += 1

                if attempt > max_retries:
                    logger.error(
                        "retry.context.max_attempts",
                        **context,
                        max_retries=max_retries,
                        error=str(e),
                        error_type=type(e).__name__,
                    )
                    raise

                # Calcular delay con jitter
                delay = RetryHelper.calculate_delay(attempt, initial_delay, max_delay)

                # Si es RateLimitError con retry_after, usarlo
                if isinstance(e, RateLimitError) and hasattr(e, "details"):
                    if e.details.get("retry_after"):
                        delay = min(e.details["retry_after"], max_delay)

                logger.info(
                    "retry.context.attempting", **context, attempt=attempt, delay=delay, error_type=type(e).__name__
                )

                if asyncio.iscoroutinefunction(func):
                    await asyncio.sleep(delay)
                else:
                    time.sleep(delay)

            except Exception as e:
                logger.error("retry.context.unexpected_error", **context, error=str(e), error_type=type(e).__name__)
                raise


# Ejemplo de uso específico para CIMA API
@exponential_backoff(max_retries=5, initial_delay=2.0, max_delay=120.0, exceptions=(RateLimitError, APIError))
async def call_cima_api_with_retry(endpoint: str, params: dict = None) -> dict:
    """
    Ejemplo de función que llama a CIMA API con retry automático.

    Args:
        endpoint: Endpoint de CIMA a llamar
        params: Parámetros de la llamada

    Returns:
        Respuesta de la API

    Raises:
        RateLimitError: Si se excede el rate limit después de reintentos
        APIError: Si hay error de API después de reintentos
    """
    # Aquí iría la lógica real de llamada a CIMA
    # Este es solo un ejemplo
    pass
