# backend/app/utils/pricing.py
"""
Utilidades de pricing farmacéutico
Funciones para conversión de precios y cálculos específicos del sector farmacéutico español
"""

import logging
from decimal import ROUND_HALF_UP, Decimal

logger = logging.getLogger(__name__)


# FUNCIÓN MOVIDA: pvp_to_pvl ahora está en external_data.nomenclator_integration
# Forma parte del procesamiento de datos oficiales, no de la lógica de negocio

from ..external_data.nomenclator_integration import pvp_to_pvl


def calculate_savings_with_discount(
    original_pvp: float,
    discount_percentage: float,
    con_recargo_equivalencia: bool = True,
) -> dict:
    """
    Calcula ahorro aplicando descuento sobre PVL y reconvirtiendo a PVP

    Args:
        original_pvp: PVP original
        discount_percentage: Porcentaje de descuento a aplicar (ej: 25.0 para 25%)
        con_recargo_equivalencia: Si aplicar recargo de equivalencia

    Returns:
        dict: Información completa de ahorro
        {
            'original_pvp': PVP original,
            'original_pvl': PVL original,
            'discounted_pvl': PVL con descuento,
            'new_pvp': Nuevo PVP tras descuento,
            'savings_pvl': Ahorro en PVL,
            'savings_pvp': Ahorro en PVP,
            'discount_applied': % descuento aplicado
        }
    """

    try:
        if original_pvp <= 0 or discount_percentage < 0 or discount_percentage > 100:
            return {
                "original_pvp": original_pvp,
                "original_pvl": 0,
                "discounted_pvl": 0,
                "new_pvp": original_pvp,
                "savings_pvl": 0,
                "savings_pvp": 0,
                "discount_applied": 0,
            }

        # Convertir a PVL
        original_pvl = pvp_to_pvl(original_pvp, con_recargo_equivalencia)

        # Aplicar descuento sobre PVL
        discount_factor = Decimal(str(discount_percentage / 100))
        pvl_decimal = Decimal(str(original_pvl))
        discounted_pvl = pvl_decimal * (1 - discount_factor)

        # Reconvertir a PVP
        new_pvp = pvl_to_pvp(float(discounted_pvl), con_recargo_equivalencia)

        return {
            "original_pvp": round(original_pvp, 2),
            "original_pvl": round(original_pvl, 2),
            "discounted_pvl": round(float(discounted_pvl), 2),
            "new_pvp": round(new_pvp, 2),
            "savings_pvl": round(original_pvl - float(discounted_pvl), 2),
            "savings_pvp": round(original_pvp - new_pvp, 2),
            "discount_applied": discount_percentage,
        }

    except Exception as e:
        logger.error(f"Error calculando ahorro con descuento: {str(e)}")
        return {
            "original_pvp": original_pvp,
            "original_pvl": 0,
            "discounted_pvl": 0,
            "new_pvp": original_pvp,
            "savings_pvl": 0,
            "savings_pvp": 0,
            "discount_applied": 0,
        }


def pvl_to_pvp(
    pvl: float,
    con_recargo_equivalencia: bool = True,
    iva_rate: float = 4.0,
    recargo_equivalencia_rate: float = 0.52,
) -> float:
    """
    Convierte PVL a PVP con IVA

    Args:
        pvl: Precio Venta Laboratorio
        con_recargo_equivalencia: Si aplicar recargo de equivalencia
        iva_rate: Tasa de IVA
        recargo_equivalencia_rate: Tasa recargo equivalencia

    Returns:
        float: PVP con IVA incluido
    """

    try:
        if pvl <= 0:
            return 0.0

        pvl_decimal = Decimal(str(pvl))
        iva_decimal = Decimal(str(iva_rate / 100))

        # Aplicar recargo de equivalencia si corresponde
        if con_recargo_equivalencia:
            recargo_decimal = Decimal(str(recargo_equivalencia_rate / 100))
            base_con_recargo = pvl_decimal * (1 + recargo_decimal)
        else:
            base_con_recargo = pvl_decimal

        # Aplicar IVA
        pvp_con_iva = base_con_recargo * (1 + iva_decimal)

        # Redondear a 2 decimales
        pvp_rounded = pvp_con_iva.quantize(Decimal("0.01"), rounding=ROUND_HALF_UP)

        return float(pvp_rounded)

    except (ValueError, TypeError) as e:
        logger.error(f"Error en conversión PVL a PVP: {pvl}, error: {str(e)}")
        return 0.0


def validate_price_calculation(original_pvp: float, calculated_pvp: float, tolerance: float = 0.02) -> bool:
    """
    Valida que un cálculo de precio sea correcto dentro de una tolerancia

    Args:
        original_pvp: PVP original
        calculated_pvp: PVP calculado tras conversiones
        tolerance: Tolerancia permitida en euros

    Returns:
        bool: True si la diferencia está dentro de la tolerancia
    """

    return abs(original_pvp - calculated_pvp) <= tolerance


# Constantes para el sector farmacéutico español
FARMACIA_IVA_RATE = 4.0  # IVA reducido medicamentos
FARMACIA_RECARGO_EQUIVALENCIA = 0.52  # Recargo equivalencia típico
PVP_ROUND_DECIMALS = 2  # Decimales estándar para precios farmacéuticos
