# backend/app/utils/memory_utils.py
"""
Utilidades para monitoreo de uso de memoria.

Issue: Render OOM crashes (2025-11-02)
Proporciona helpers para logging estructurado de consumo de memoria
en operaciones críticas para prevenir Out of Memory errors.
"""
import logging
import os

import psutil

logger = logging.getLogger(__name__)


def log_memory_usage(operation_name: str):
    """
    Log current memory usage for monitoring production OOM issues.

    Registra el uso de memoria del proceso actual con métricas estructuradas
    para facilitar análisis en Render/Datadog/Grafana.

    Args:
        operation_name: Nombre descriptivo de la operación (ej: "storage_usage_start")

    Example:
        >>> log_memory_usage("catalog_sync_start")
        [MEMORY] catalog_sync_start: 245.32 MB RSS

    Notes:
        - Usa psutil para obtener métricas del proceso actual
        - Silencia errores para no interrumpir funcionalidad crítica
        - Logs incluyen extra metadata para filtrado en producción
    """
    try:
        process = psutil.Process(os.getpid())
        memory_info = process.memory_info()
        memory_mb = memory_info.rss / 1024 / 1024

        logger.info(
            f"[MEMORY] {operation_name}: {memory_mb:.2f} MB RSS",
            extra={
                "operation": operation_name,
                "memory_mb": memory_mb,
                "memory_percent": process.memory_percent(),
            }
        )
    except Exception as e:
        logger.warning(f"[MEMORY] Failed to log memory for {operation_name}: {e}")
