# backend/app/utils/laboratory_queries.py
"""
LaboratoryQueryBuilder - FASE 2.1 Issue #23
Query builder utility para eliminar duplicación SQL en laboratory mapping system

Funcionalidades:
- Templates SQL reutilizables con parameterización segura
- Builder pattern para construir queries dinámicas
- Protección SQL injection built-in
- Configuración centralizada de límites y timeouts
- Optimización para indexes de product_catalog
"""

import logging
from typing import Any, Dict, List, Optional, Tuple

from sqlalchemy import text

from ..config.laboratory_config import LaboratoryConfig

logger = logging.getLogger(__name__)


class LaboratoryQueryBuilder:
    """Builder para queries SQL de laboratory mapping sin duplicación"""

    def __init__(self):
        """Inicializar builder con configuración centralizada"""
        self._config = LaboratoryConfig

    # === TEMPLATES SQL BASE (SIN DUPLICACIÓN) ===

    def _get_base_select_template(self) -> str:
        """Template base para SELECT de laboratorios"""
        return f"""
        SELECT {self._config.get_base_laboratory_columns()}
        FROM {self._config.PRODUCT_CATALOG_TABLE} pc
        """

    def _get_base_where_conditions(self, include_active_only: bool = True) -> List[str]:
        """Obtener condiciones WHERE base comunes a todas las queries"""
        conditions = self._config.get_laboratory_base_conditions()

        if include_active_only:
            conditions.append(f"pc.{self._config.get_active_laboratory_condition()}")

        return conditions

    def _build_where_clause(self, conditions: List[str]) -> str:
        """Construir cláusula WHERE desde lista de condiciones"""
        if not conditions:
            return ""

        return "WHERE " + " AND ".join(conditions)

    def _add_standard_ordering(self) -> str:
        """Ordenamiento estándar por nombre de laboratorio"""
        return f"ORDER BY pc.{self._config.NOMEN_NAME_COLUMN}"

    def _add_limit_clause(self, limit: Optional[int] = None) -> str:
        """Agregar LIMIT según configuración"""
        actual_limit = limit or self._config.LABORATORY_MAPPING_LIMIT
        return f"LIMIT {actual_limit}"

    def _add_pagination_clause(self, page: int, per_page: int) -> str:
        """Agregar LIMIT y OFFSET para paginación"""
        offset = (page - 1) * per_page
        return f"LIMIT {per_page} OFFSET {offset}"

    def _add_cursor_based_clause(
        self, cursor_id: Optional[str], per_page: int, direction: str = "next"
    ) -> Tuple[str, Dict[str, Any]]:
        """Agregar WHERE para cursor-based pagination (performance en páginas altas)"""
        params = {}
        clause = f"LIMIT {per_page}"

        if cursor_id:
            if direction == "next":
                clause = f"WHERE pc.{self._config.NOMEN_CODE_COLUMN} > :cursor_id {clause}"
            else:  # previous
                clause = f"WHERE pc.{self._config.NOMEN_CODE_COLUMN} < :cursor_id {clause}"

            params["cursor_id"] = cursor_id

        return clause, params

    def _build_count_query(self, base_conditions: List[str]) -> text:
        """Construir query COUNT para obtener total de elementos"""
        query_parts = [
            f"SELECT COUNT(DISTINCT pc.{self._config.NOMEN_CODE_COLUMN}) as total",
            f"FROM {self._config.PRODUCT_CATALOG_TABLE} pc",
            self._build_where_clause(base_conditions),
        ]

        query_sql = " ".join(filter(None, query_parts))
        return text(query_sql)

    # === BUILDERS ESPECÍFICOS (REUSANDO TEMPLATES) ===

    def build_codes_to_names_query(
        self,
        codes: Optional[List[str]] = None,
        page: Optional[int] = None,
        per_page: Optional[int] = None,
        use_cursor: bool = False,
        cursor_id: Optional[str] = None,
    ) -> Tuple[text, Dict[str, Any]]:
        """
        Construir query para mapeo códigos → nombres
        Elimina duplicación de la query en _fetch_codes_to_names_from_db
        """
        base_conditions = self._get_base_where_conditions(include_active_only=True)
        params = {}

        if codes:
            # Query específica para códigos solicitados
            code_conditions = []
            for i, code in enumerate(codes):
                if code and code.isdigit() and len(code) <= self._config.MAX_LAB_CODE_LENGTH:
                    param_name = f"code_{i}"
                    params[param_name] = code
                    code_conditions.append(f"pc.{self._config.NOMEN_CODE_COLUMN} = :{param_name}")

            if not code_conditions:
                # No hay códigos válidos - query vacía
                return text("SELECT NULL as nomen_codigo_laboratorio, NULL as nomen_laboratorio WHERE FALSE"), {}

            # Agregar condiciones OR para códigos específicos
            specific_condition = f"({' OR '.join(code_conditions)})"
            base_conditions.append(specific_condition)
        else:
            # Query para todos los laboratorios activos con límite
            params["limit_val"] = self._config.LABORATORY_MAPPING_LIMIT

        # Construir query completa
        query_parts = [
            self._get_base_select_template(),
            self._build_where_clause(base_conditions),
            self._add_standard_ordering(),
        ]

        # FASE 2.2: Añadir paginación si está especificada
        if page is not None and per_page is not None:
            if use_cursor and cursor_id:
                # Cursor-based pagination para mejor performance
                cursor_clause, cursor_params = self._add_cursor_based_clause(cursor_id, per_page)
                query_parts.append(cursor_clause)
                params.update(cursor_params)
            else:
                # Paginación tradicional con OFFSET
                query_parts.append(self._add_pagination_clause(page, per_page))
        elif not codes:  # Solo agregar LIMIT si no hay códigos específicos
            query_parts.append("LIMIT :limit_val")

        query_sql = " ".join(query_parts)
        return text(query_sql), params

    def build_names_to_codes_query(
        self,
        names: Optional[List[str]] = None,
        page: Optional[int] = None,
        per_page: Optional[int] = None,
        use_cursor: bool = False,
        cursor_id: Optional[str] = None,
    ) -> Tuple[text, Dict[str, Any]]:
        """
        Construir query para mapeo nombres → códigos
        Elimina duplicación de la query en _fetch_names_to_codes_from_db
        """
        base_conditions = self._get_base_where_conditions(include_active_only=True)
        params = {}

        if names:
            # Query específica para nombres solicitados
            name_conditions = []
            for i, name in enumerate(names):
                if name and len(name) <= self._config.MAX_NAME_LENGTH:
                    param_name = f"name_{i}"
                    params[param_name] = f"%{name}%"
                    name_conditions.append(f"UPPER(pc.{self._config.NOMEN_NAME_COLUMN}) LIKE UPPER(:{param_name})")

            if not name_conditions:
                # No hay nombres válidos - query vacía
                return text("SELECT NULL as nomen_codigo_laboratorio, NULL as nomen_laboratorio WHERE FALSE"), {}

            # Agregar condiciones OR para nombres específicos
            specific_condition = f"({' OR '.join(name_conditions)})"
            base_conditions.append(specific_condition)
        else:
            # Query para todos los laboratorios activos con límite
            params["limit_val"] = self._config.LABORATORY_MAPPING_LIMIT

        # Construir query completa
        query_parts = [
            self._get_base_select_template(),
            self._build_where_clause(base_conditions),
            self._add_standard_ordering(),
        ]

        # FASE 2.2: Añadir paginación si está especificada
        if page is not None and per_page is not None:
            if use_cursor and cursor_id:
                # Cursor-based pagination para mejor performance
                cursor_clause, cursor_params = self._add_cursor_based_clause(cursor_id, per_page)
                query_parts.append(cursor_clause)
                params.update(cursor_params)
            else:
                # Paginación tradicional con OFFSET
                query_parts.append(self._add_pagination_clause(page, per_page))
        elif not names:  # Solo agregar LIMIT si no hay nombres específicos
            query_parts.append("LIMIT :limit_val")

        query_sql = " ".join(query_parts)
        return text(query_sql), params

    def build_generic_laboratories_query(
        self,
        page: int = 1,
        per_page: int = 50,
        search: Optional[str] = None,
        use_cursor: bool = False,
        cursor_id: Optional[str] = None,
    ) -> Tuple[text, Dict[str, Any]]:
        """
        Construir query para laboratorios genéricos con paginación
        Elimina duplicación de la query en _fetch_generic_laboratories_from_db
        """
        base_conditions = self._get_base_where_conditions(include_active_only=True)
        params = {}

        # Agregar condiciones para laboratorios genéricos
        generic_patterns = self._config.get_generic_laboratory_patterns()
        pattern_conditions = []

        for i, pattern in enumerate(generic_patterns):
            param_name = f"pattern_{i}"
            params[param_name] = f"%{pattern}%"
            pattern_conditions.append(f"UPPER(pc.{self._config.NOMEN_NAME_COLUMN}) LIKE UPPER(:{param_name})")

        if pattern_conditions:
            generic_condition = f"({' OR '.join(pattern_conditions)})"
            base_conditions.append(generic_condition)

        # Agregar filtro de búsqueda si está presente
        if search:
            search_condition = f"UPPER(pc.{self._config.NOMEN_NAME_COLUMN}) LIKE UPPER(:search_term)"
            base_conditions.append(search_condition)
            params["search_term"] = f"%{search}%"

        # FASE 2.2: Paginación flexible (traditional vs cursor-based)
        query_parts = [
            self._get_base_select_template(),
            self._build_where_clause(base_conditions),
            self._add_standard_ordering(),
        ]

        if use_cursor and cursor_id:
            # Cursor-based pagination para mejor performance en páginas altas
            cursor_clause, cursor_params = self._add_cursor_based_clause(cursor_id, per_page)
            query_parts.append(cursor_clause)
            params.update(cursor_params)
        else:
            # Paginación tradicional con OFFSET
            offset = (page - 1) * per_page
            params.update({"limit_val": per_page, "offset_val": offset})
            query_parts.append("LIMIT :limit_val OFFSET :offset_val")

        query_sql = " ".join(query_parts)
        return text(query_sql), params

    # === QUERY ANALYSIS Y OPTIMIZATION ===

    def analyze_query_performance(self, query: text, params: Dict[str, Any]) -> Dict[str, Any]:
        """Analizar performance potencial de una query (para debugging)"""
        query_str = str(query)

        analysis = {
            "uses_indexes": False,
            "has_limit": "LIMIT" in query_str,
            "has_order_by": "ORDER BY" in query_str,
            "parameter_count": len(params),
            "estimated_complexity": "LOW",  # LOW, MEDIUM, HIGH
        }

        # Detectar uso de indexes principales
        if self._config.NOMEN_CODE_COLUMN in query_str:
            analysis["uses_indexes"] = True

        # Estimar complejidad basado en condiciones
        condition_count = query_str.count("AND") + query_str.count("OR")
        if condition_count > 5:
            analysis["estimated_complexity"] = "HIGH"
        elif condition_count > 2:
            analysis["estimated_complexity"] = "MEDIUM"

        return analysis

    def build_count_query_for_codes(self, codes: Optional[List[str]] = None) -> Tuple[text, Dict[str, Any]]:
        """Construir query de conteo para codes_to_names"""
        base_conditions = self._get_base_where_conditions(include_active_only=True)
        params = {}

        if codes:
            code_conditions = []
            for i, code in enumerate(codes):
                if code and code.isdigit() and len(code) <= self._config.MAX_LAB_CODE_LENGTH:
                    param_name = f"code_{i}"
                    params[param_name] = code
                    code_conditions.append(f"pc.{self._config.NOMEN_CODE_COLUMN} = :{param_name}")

            if code_conditions:
                specific_condition = f"({' OR '.join(code_conditions)})"
                base_conditions.append(specific_condition)

        return self._build_count_query(base_conditions), params

    def build_count_query_for_names(self, names: Optional[List[str]] = None) -> Tuple[text, Dict[str, Any]]:
        """Construir query de conteo para names_to_codes"""
        base_conditions = self._get_base_where_conditions(include_active_only=True)
        params = {}

        if names:
            name_conditions = []
            for i, name in enumerate(names):
                if name and len(name) <= self._config.MAX_NAME_LENGTH:
                    param_name = f"name_{i}"
                    params[param_name] = f"%{name}%"
                    name_conditions.append(f"UPPER(pc.{self._config.NOMEN_NAME_COLUMN}) LIKE UPPER(:{param_name})")

            if name_conditions:
                specific_condition = f"({' OR '.join(name_conditions)})"
                base_conditions.append(specific_condition)

        return self._build_count_query(base_conditions), params

    def build_count_query_for_generics(self, search: Optional[str] = None) -> Tuple[text, Dict[str, Any]]:
        """Construir query de conteo para generic_laboratories"""
        base_conditions = self._get_base_where_conditions(include_active_only=True)
        params = {}

        # Agregar condiciones para laboratorios genéricos
        generic_patterns = self._config.get_generic_laboratory_patterns()
        pattern_conditions = []

        for i, pattern in enumerate(generic_patterns):
            param_name = f"pattern_{i}"
            params[param_name] = f"%{pattern}%"
            pattern_conditions.append(f"UPPER(pc.{self._config.NOMEN_NAME_COLUMN}) LIKE UPPER(:{param_name})")

        if pattern_conditions:
            generic_condition = f"({' OR '.join(pattern_conditions)})"
            base_conditions.append(generic_condition)

        # Agregar filtro de búsqueda si está presente
        if search:
            search_condition = f"UPPER(pc.{self._config.NOMEN_NAME_COLUMN}) LIKE UPPER(:search_term)"
            base_conditions.append(search_condition)
            params["search_term"] = f"%{search}%"

        return self._build_count_query(base_conditions), params

    def get_recommended_indexes(self) -> List[str]:
        """Obtener lista de indexes recomendados para laboratory queries"""
        return [
            f"CREATE INDEX IF NOT EXISTS idx_product_catalog_nomen_codigo ON {self._config.PRODUCT_CATALOG_TABLE} ({self._config.NOMEN_CODE_COLUMN})",
            f"CREATE INDEX IF NOT EXISTS idx_product_catalog_nomen_laboratorio ON {self._config.PRODUCT_CATALOG_TABLE} ({self._config.NOMEN_NAME_COLUMN})",
            f"CREATE INDEX IF NOT EXISTS idx_product_catalog_nomen_estado ON {self._config.PRODUCT_CATALOG_TABLE} ({self._config.NOMEN_STATUS_COLUMN})",
            f"CREATE INDEX IF NOT EXISTS idx_product_catalog_nomen_composite ON {self._config.PRODUCT_CATALOG_TABLE} ({self._config.NOMEN_CODE_COLUMN}, {self._config.NOMEN_NAME_COLUMN}, {self._config.NOMEN_STATUS_COLUMN})",
        ]

    # === VALIDATION Y SECURITY ===

    def validate_query_parameters(self, params: Dict[str, Any]) -> bool:
        """Validar que los parámetros de query son seguros"""
        for key, value in params.items():
            # Verificar tipos seguros
            if not isinstance(value, (str, int, float, type(None))):
                logger.warning(f"[QUERY_BUILDER] Parámetro inseguro detectado: {key} = {type(value)}")
                return False

            # Verificar strings no contienen SQL injection
            if isinstance(value, str):
                dangerous_patterns = ["';", "--", "/*", "*/", "xp_", "sp_"]
                for pattern in dangerous_patterns:
                    if pattern.lower() in value.lower():
                        logger.warning(f"[QUERY_BUILDER] Posible SQL injection en parámetro: {key}")
                        return False

        return True

    def should_use_cursor_pagination(self, page: int, total_items: int) -> bool:
        """Determinar si usar cursor-based pagination basado en página y dataset"""
        # Usar cursor-based para páginas altas o datasets grandes
        from ..schemas.laboratory_validation import LaboratoryValidationConstants

        return page > LaboratoryValidationConstants.CURSOR_BASED_THRESHOLD or total_items > 10000

    def log_query_execution(self, query_type: str, params: Dict[str, Any], execution_time: float):
        """Log detallado de ejecución de query para monitoreo"""
        logger.info(
            f"[QUERY_BUILDER] Query ejecutada: type={query_type}, "
            f"params_count={len(params)}, execution_time={execution_time:.3f}s"
        )

        if execution_time > 1.0:  # Queries lentas > 1s
            logger.warning(
                f"[QUERY_BUILDER] Query lenta detectada: type={query_type}, "
                f"time={execution_time:.3f}s, params={list(params.keys())[:5]}, "  # Primeros 5 keys
                f"pagination={('offset_val' in params or 'cursor_id' in params)}"
            )


class PaginationConfig:
    """Configuración para estrategias de paginación"""

    CURSOR_THRESHOLD_PAGE = 100  # Usar cursor después de página 100
    CURSOR_THRESHOLD_ITEMS = 10000  # Usar cursor para datasets > 10k
    MAX_OFFSET_ITEMS = 50000  # Máximo para OFFSET, después cursor obligatorio
    COUNT_ESTIMATE_THRESHOLD = 100000  # Usar COUNT estimado para datasets enormes


# Instancia global del query builder
laboratory_query_builder = LaboratoryQueryBuilder()
