"""
Helper centralizado para logging estructurado de errores con traceback completo.
Implementado según PR #269 para evitar duplicación de código.
"""

import traceback
from logging import Logger
from typing import Any, Optional


def log_structured_error(
    logger: Logger, exception: Optional[Exception], context_message: str, **additional_context: Any
) -> None:
    """
    Registra un error de forma estructurada con traceback completo.

    Este helper centraliza el patrón de logging de errores implementado en PR #269,
    capturando automáticamente el mensaje de error, tipo, y traceback completo,
    junto con cualquier contexto adicional proporcionado.

    Args:
        logger: Logger instance para registrar el error
        exception: Excepción capturada a registrar
        context_message: Mensaje descriptivo del contexto donde ocurrió el error
        **additional_context: Campos adicionales para el contexto del log
            (ej: phase, batch_size, retry_count, correlation_id, etc.)

    Ejemplo de uso:
        ```python
        try:
            # Código que puede fallar
            result = risky_operation()
        except Exception as e:
            log_structured_error(
                logger=logger,
                exception=e,
                context_message="Error en sincronización CIMA",
                phase="cima_sync",
                batch_size=300,
                current_index=1500,
                total_items=67000
            )
            raise
        ```

    Casos de uso comunes:
        - Sincronización CIMA con contexto de batch processing
        - Errores en catalog maintenance con estado del proceso
        - Fallos en enrichment con información del producto
        - Excepciones en parsers con detalles del archivo
        - Errores de API con información de la request

    El log resultante incluirá:
        - error: Mensaje de la excepción
        - error_type: Nombre de la clase de excepción
        - traceback: Stack trace completo
        - exc_info: True para incluir información de excepción
        - Cualquier campo adicional pasado en **additional_context

    Nota: Los campos base (error, error_type, traceback, exc_info) no pueden
    ser sobrescritos por additional_context para mantener integridad del log.
    """
    # Manejar caso de excepción None
    if exception is None:
        error_msg = "None"
        error_type = "NoneType"
        error_traceback = "No exception provided"
    else:
        # Extraer información de la excepción
        error_msg = str(exception)
        error_type = type(exception).__name__
        # Si hay un traceback activo, usarlo. Si no, generar uno mínimo
        current_tb = traceback.format_exc()
        if current_tb == "NoneType: None\n":
            # No hay traceback activo, crear uno mínimo
            error_traceback = f"Direct exception logging: {error_type}: {error_msg}"
        else:
            error_traceback = current_tb

    # Construir campos estructurados base
    structured_fields = {"error": error_msg, "error_type": error_type, "traceback": error_traceback}

    # Agregar contexto adicional (sin sobrescribir campos base)
    for key, value in additional_context.items():
        if key not in structured_fields and key != "exc_info":
            structured_fields[key] = value

    # Para el logger estándar de Python, usar 'extra' para campos estructurados
    # y exc_info como parámetro separado
    logger.error(context_message, exc_info=True, extra=structured_fields)
