"""
Utilidades para manejo de fechas y horas en timezone UTC.

Python 3.13 depreca datetime.utcnow(), este módulo provee
funciones compatibles usando datetime.now(timezone.utc).
"""

from datetime import datetime, timezone


def utc_now() -> datetime:
    """
    Retorna el datetime actual en UTC timezone-aware.

    Reemplazo para datetime.utcnow() deprecado en Python 3.13.

    Returns:
        datetime: Timestamp actual en UTC con tzinfo=timezone.utc

    Examples:
        >>> from app.utils.datetime_utils import utc_now
        >>> timestamp = utc_now()
        >>> timestamp.tzinfo == timezone.utc
        True
    """
    return datetime.now(timezone.utc)


def make_aware(dt: datetime) -> datetime:
    """
    Convierte un datetime naive (sin timezone) a timezone-aware en UTC.

    Si el datetime ya es aware (tiene tzinfo), lo retorna sin modificar.
    Si es naive, asume que representa UTC y le agrega tzinfo=timezone.utc.

    Args:
        dt: datetime a convertir (puede ser naive o aware)

    Returns:
        datetime: Timestamp con tzinfo=timezone.utc

    Examples:
        >>> from datetime import datetime
        >>> naive_dt = datetime(2025, 10, 26, 10, 30, 0)
        >>> aware_dt = make_aware(naive_dt)
        >>> aware_dt.tzinfo == timezone.utc
        True

        >>> already_aware = datetime.now(timezone.utc)
        >>> result = make_aware(already_aware)
        >>> result is already_aware  # Sin modificar
        True

    Notes:
        Esta función es un quick fix para compatibilidad con SQLAlchemy
        que retorna datetimes naive cuando Column(DateTime) no especifica
        timezone=True.

        Long-term fix: Migrar modelos a Column(DateTime(timezone=True))
    """
    if dt.tzinfo is None:
        # Naive datetime - asumir UTC y agregar tzinfo
        return dt.replace(tzinfo=timezone.utc)
    else:
        # Ya es aware - retornar sin modificar
        return dt
