# backend/app/services/word_action_plan_generator.py
"""
Generador de Documento Word para Planes de Acción (Issue #513).

Usa python-docx para generar un documento estructurado:
- Portada: Farmacia, período, fecha
- Resumen Ejecutivo: KPIs y oportunidad total
- Acciones por Prioridad: Tablas agrupadas
- Checklist de Seguimiento: Tabla para marcar progreso
"""

import io
import structlog
from datetime import date
from typing import Optional

from docx import Document
from docx.shared import Inches, Pt, Cm, RGBColor
from docx.enum.text import WD_ALIGN_PARAGRAPH
from docx.enum.table import WD_TABLE_ALIGNMENT
from docx.enum.style import WD_STYLE_TYPE
from docx.oxml.ns import qn
from docx.oxml import OxmlElement

from app.schemas.action_plans import (
    ActionPlanData,
    ActionPlanItem,
    ActionPriority,
)

logger = structlog.get_logger(__name__)

# Colores corporativos (RGB)
COLORS = {
    "primary": RGBColor(44, 62, 80),       # Dark blue (#2c3e50)
    "success": RGBColor(39, 174, 96),      # Green (#27ae60)
    "danger": RGBColor(231, 76, 60),       # Red (#e74c3c)
    "warning": RGBColor(243, 156, 18),     # Orange (#f39c12)
    "critical": RGBColor(142, 68, 173),    # Purple (#8e44ad)
    "muted": RGBColor(127, 140, 141),      # Gray (#7f8c8d)
    "light": RGBColor(236, 240, 241),      # Light gray (#ecf0f1)
    "white": RGBColor(255, 255, 255),
}

PRIORITY_COLORS = {
    ActionPriority.CRITICAL: COLORS["critical"],
    ActionPriority.HIGH: COLORS["danger"],
    ActionPriority.MEDIUM: COLORS["warning"],
    ActionPriority.LOW: COLORS["success"],
}

PRIORITY_LABELS = {
    ActionPriority.CRITICAL: "CRITICO - Actuar Inmediatamente",
    ActionPriority.HIGH: "ALTA - Esta Semana",
    ActionPriority.MEDIUM: "MEDIA - Este Mes",
    ActionPriority.LOW: "BAJA - Cuando Sea Posible",
}


class WordActionPlanGenerator:
    """
    Genera documento Word con plan de acción.

    Estructura del documento:
    1. Portada (1 página)
    2. Resumen Ejecutivo (1 página)
    3. Acciones por Prioridad (n páginas)
    4. Checklist de Seguimiento (1 página)
    """

    def __init__(self):
        self.doc: Optional[Document] = None

    def generate(self, data: ActionPlanData) -> bytes:
        """
        Genera documento Word.

        Args:
            data: ActionPlanData con todos los datos

        Returns:
            bytes del documento Word
        """
        logger.info(
            "word_action_plan_generator.generate.start",
            pharmacy=data.pharmacy_name,
            actions_count=len(data.actions),
        )

        self.doc = Document()
        self._setup_styles()

        # 1. Portada
        self._build_cover(data)

        # 2. Resumen Ejecutivo
        self._build_executive_summary(data)

        # 3. Acciones por Prioridad
        self._build_actions_by_priority(data)

        # 4. Checklist de Seguimiento
        self._build_checklist(data)

        # Guardar en buffer
        buffer = io.BytesIO()
        self.doc.save(buffer)
        buffer.seek(0)

        logger.info(
            "word_action_plan_generator.generate.complete",
            file_size_kb=len(buffer.getvalue()) / 1024,
        )

        return buffer.getvalue()

    def _setup_styles(self):
        """Configurar estilos del documento."""
        styles = self.doc.styles

        # Estilo para título principal
        if "ActionTitle" not in [s.name for s in styles]:
            title_style = styles.add_style("ActionTitle", WD_STYLE_TYPE.PARAGRAPH)
            title_style.font.size = Pt(28)
            title_style.font.bold = True
            title_style.font.color.rgb = COLORS["primary"]
            title_style.paragraph_format.alignment = WD_ALIGN_PARAGRAPH.CENTER
            title_style.paragraph_format.space_after = Pt(12)

        # Estilo para subtítulo
        if "ActionSubtitle" not in [s.name for s in styles]:
            subtitle_style = styles.add_style("ActionSubtitle", WD_STYLE_TYPE.PARAGRAPH)
            subtitle_style.font.size = Pt(14)
            subtitle_style.font.color.rgb = COLORS["muted"]
            subtitle_style.paragraph_format.alignment = WD_ALIGN_PARAGRAPH.CENTER
            subtitle_style.paragraph_format.space_after = Pt(6)

        # Estilo para header de sección
        if "SectionHeader" not in [s.name for s in styles]:
            section_style = styles.add_style("SectionHeader", WD_STYLE_TYPE.PARAGRAPH)
            section_style.font.size = Pt(16)
            section_style.font.bold = True
            section_style.font.color.rgb = COLORS["primary"]
            section_style.paragraph_format.space_before = Pt(18)
            section_style.paragraph_format.space_after = Pt(12)

    def _build_cover(self, data: ActionPlanData):
        """Construir portada."""
        # Espacio inicial
        for _ in range(5):
            self.doc.add_paragraph()

        # Título principal
        title = self.doc.add_paragraph("PLAN DE ACCION", style="ActionTitle")

        # Subtítulo
        self.doc.add_paragraph("Parafarmacia - Mejora Continua", style="ActionSubtitle")

        # Espacio
        for _ in range(3):
            self.doc.add_paragraph()

        # Datos de la farmacia
        info = self.doc.add_paragraph()
        info.alignment = WD_ALIGN_PARAGRAPH.CENTER

        # Nombre farmacia
        run = info.add_run(f"{data.pharmacy_name}\n")
        run.font.size = Pt(18)
        run.font.bold = True
        run.font.color.rgb = COLORS["primary"]

        # Período
        period_str = self._format_period(data.period_start, data.period_end)
        run = info.add_run(f"\n{period_str}\n")
        run.font.size = Pt(14)
        run.font.color.rgb = COLORS["muted"]

        # Oportunidad total
        if data.summary.total_economic_opportunity > 0:
            self.doc.add_paragraph()
            opp = self.doc.add_paragraph()
            opp.alignment = WD_ALIGN_PARAGRAPH.CENTER
            run = opp.add_run("Oportunidad Identificada\n")
            run.font.size = Pt(12)
            run.font.color.rgb = COLORS["muted"]
            run = opp.add_run(f"{self._format_currency(data.summary.total_economic_opportunity)}")
            run.font.size = Pt(32)
            run.font.bold = True
            run.font.color.rgb = COLORS["success"]

        # Footer de portada
        for _ in range(5):
            self.doc.add_paragraph()

        footer = self.doc.add_paragraph()
        footer.alignment = WD_ALIGN_PARAGRAPH.CENTER
        run = footer.add_run(f"Generado: {data.generated_at[:16].replace('T', ' ')}")
        run.font.size = Pt(10)
        run.font.color.rgb = COLORS["muted"]
        run = footer.add_run(f"\n{data.insights_source}")
        run.font.size = Pt(9)
        run.font.color.rgb = COLORS["muted"]

        # Salto de página
        self.doc.add_page_break()

    def _build_executive_summary(self, data: ActionPlanData):
        """Construir resumen ejecutivo."""
        self.doc.add_paragraph("RESUMEN EJECUTIVO", style="SectionHeader")

        summary = data.summary

        # Tabla de resumen
        table = self.doc.add_table(rows=2, cols=4)
        table.alignment = WD_TABLE_ALIGNMENT.CENTER
        self._set_table_borders(table)

        # Header row
        header_cells = table.rows[0].cells
        headers = ["Total Acciones", "Criticas", "Alta Prioridad", "Oportunidad Total"]
        for i, header in enumerate(headers):
            header_cells[i].text = header
            self._style_header_cell(header_cells[i])

        # Data row
        data_cells = table.rows[1].cells
        values = [
            str(summary.total_actions),
            str(summary.critical_count),
            str(summary.high_count),
            self._format_currency(summary.total_economic_opportunity),
        ]
        for i, value in enumerate(values):
            data_cells[i].text = value
            self._style_data_cell(data_cells[i], center=True, bold=True)

        self.doc.add_paragraph()

        # Distribución por categoría
        if summary.by_category:
            self.doc.add_paragraph("Acciones por Categoria:", style="Heading 2")
            for cat, count in summary.by_category.items():
                p = self.doc.add_paragraph(style="List Bullet")
                p.add_run(f"{cat.capitalize()}: ").bold = True
                p.add_run(f"{count} acciones")

        self.doc.add_paragraph()

        # Recomendación principal
        if summary.main_recommendation:
            rec_para = self.doc.add_paragraph()
            run = rec_para.add_run("Recomendacion Principal: ")
            run.bold = True
            run.font.color.rgb = COLORS["primary"]
            rec_para.add_run(summary.main_recommendation)

        self.doc.add_page_break()

    def _build_actions_by_priority(self, data: ActionPlanData):
        """Construir sección de acciones agrupadas por prioridad."""
        self.doc.add_paragraph("PLAN DE ACCIONES DETALLADO", style="SectionHeader")

        # Agrupar acciones por prioridad
        actions_by_priority = {
            ActionPriority.CRITICAL: [],
            ActionPriority.HIGH: [],
            ActionPriority.MEDIUM: [],
            ActionPriority.LOW: [],
        }

        for action in data.actions:
            actions_by_priority[action.priority].append(action)

        # Generar sección para cada prioridad con acciones
        for priority in [
            ActionPriority.CRITICAL,
            ActionPriority.HIGH,
            ActionPriority.MEDIUM,
            ActionPriority.LOW,
        ]:
            actions = actions_by_priority[priority]
            if not actions:
                continue

            self._build_priority_section(priority, actions)

    def _build_priority_section(self, priority: ActionPriority, actions: list[ActionPlanItem]):
        """Construir sección para una prioridad específica."""
        # Header de prioridad
        header = self.doc.add_paragraph()
        run = header.add_run(f"  {PRIORITY_LABELS[priority]}  ")
        run.font.size = Pt(14)
        run.font.bold = True
        run.font.color.rgb = RGBColor(255, 255, 255)

        # Simular background (limitación de python-docx)
        # Añadimos una línea coloreada debajo
        line = self.doc.add_paragraph()
        run = line.add_run("_" * 80)
        run.font.color.rgb = PRIORITY_COLORS[priority]

        # Tabla de acciones
        for i, action in enumerate(actions, 1):
            self._build_action_card(action, i)

        self.doc.add_paragraph()

    def _build_action_card(self, action: ActionPlanItem, index: int):
        """Construir card para una acción individual."""
        # Título de la acción
        title_para = self.doc.add_paragraph()
        run = title_para.add_run(f"{index}. {action.title}")
        run.font.size = Pt(12)
        run.font.bold = True
        run.font.color.rgb = COLORS["primary"]

        # Impacto económico
        impact_para = self.doc.add_paragraph()
        run = impact_para.add_run("Impacto estimado: ")
        run.font.size = Pt(10)
        run = impact_para.add_run(action.economic_impact)
        run.font.size = Pt(10)
        run.font.bold = True
        run.font.color.rgb = COLORS["success"]

        # Descripción
        if action.description:
            desc_para = self.doc.add_paragraph()
            desc_para.add_run(action.description).font.size = Pt(10)

        # Pasos de acción
        if action.action_steps:
            steps_header = self.doc.add_paragraph()
            run = steps_header.add_run("Pasos a seguir:")
            run.font.size = Pt(10)
            run.bold = True

            for step in action.action_steps:
                step_para = self.doc.add_paragraph(style="List Bullet")
                step_para.add_run(step).font.size = Pt(10)

        # Productos afectados (si hay)
        if action.affected_count > 0:
            affected_para = self.doc.add_paragraph()
            run = affected_para.add_run(f"Productos afectados: {action.affected_count}")
            run.font.size = Pt(10)
            run.font.italic = True
            run.font.color.rgb = COLORS["muted"]

            if action.top_affected:
                top_para = self.doc.add_paragraph()
                run = top_para.add_run("Ejemplos: ")
                run.font.size = Pt(9)
                run.font.color.rgb = COLORS["muted"]
                run = top_para.add_run(", ".join(action.top_affected[:3]))
                run.font.size = Pt(9)
                run.font.color.rgb = COLORS["muted"]

        # Tabla de seguimiento inline
        tracking_table = self.doc.add_table(rows=1, cols=3)
        tracking_table.alignment = WD_TABLE_ALIGNMENT.LEFT
        cells = tracking_table.rows[0].cells

        # Responsable
        cells[0].text = "Responsable: ______________"
        cells[0].paragraphs[0].runs[0].font.size = Pt(9)

        # Fecha límite
        deadline_text = action.suggested_deadline or "_______________"
        cells[1].text = f"Plazo: {deadline_text}"
        cells[1].paragraphs[0].runs[0].font.size = Pt(9)

        # Estado
        cells[2].text = "Estado: [ ] Pendiente [ ] Completado"
        cells[2].paragraphs[0].runs[0].font.size = Pt(9)

        # Separador
        self.doc.add_paragraph()

    def _build_checklist(self, data: ActionPlanData):
        """Construir checklist de seguimiento final."""
        self.doc.add_page_break()
        self.doc.add_paragraph("CHECKLIST DE SEGUIMIENTO", style="SectionHeader")

        intro = self.doc.add_paragraph()
        intro.add_run("Utilice esta tabla para hacer seguimiento del progreso de las acciones:").font.size = Pt(10)

        self.doc.add_paragraph()

        # Tabla de checklist
        table = self.doc.add_table(rows=len(data.actions) + 1, cols=5)
        table.alignment = WD_TABLE_ALIGNMENT.CENTER
        self._set_table_borders(table)

        # Header
        headers = ["#", "Accion", "Prioridad", "Responsable", "Estado"]
        header_row = table.rows[0].cells
        for i, header in enumerate(headers):
            header_row[i].text = header
            self._style_header_cell(header_row[i])

        # Filas de datos
        for i, action in enumerate(data.actions):
            row = table.rows[i + 1].cells
            row[0].text = str(i + 1)
            row[1].text = action.title[:40] + ("..." if len(action.title) > 40 else "")
            row[2].text = action.priority.value.capitalize()
            row[3].text = ""  # Campo vacío para rellenar
            row[4].text = "[ ]"  # Checkbox vacío

            for cell in row:
                self._style_data_cell(cell, font_size=9)

        self.doc.add_paragraph()

        # Notas finales
        notes = self.doc.add_paragraph()
        run = notes.add_run("Notas:")
        run.font.bold = True
        self.doc.add_paragraph("_" * 70)
        self.doc.add_paragraph("_" * 70)
        self.doc.add_paragraph("_" * 70)

        # Footer
        self.doc.add_paragraph()
        footer = self.doc.add_paragraph()
        footer.alignment = WD_ALIGN_PARAGRAPH.CENTER
        run = footer.add_run(f"Documento generado por xFarma | {data.generated_at[:10]}")
        run.font.size = Pt(9)
        run.font.color.rgb = COLORS["muted"]

    # =========================================================================
    # Helpers de estilo de tablas
    # =========================================================================

    def _set_table_borders(self, table):
        """Añadir bordes a tabla."""
        tbl = table._tbl
        tblPr = tbl.tblPr if tbl.tblPr is not None else OxmlElement("w:tblPr")

        tblBorders = OxmlElement("w:tblBorders")
        for border_name in ["top", "left", "bottom", "right", "insideH", "insideV"]:
            border = OxmlElement(f"w:{border_name}")
            border.set(qn("w:val"), "single")
            border.set(qn("w:sz"), "4")
            border.set(qn("w:color"), "CCCCCC")
            tblBorders.append(border)

        tblPr.append(tblBorders)
        if tbl.tblPr is None:
            tbl.insert(0, tblPr)

    def _style_header_cell(self, cell):
        """Estilizar celda de header."""
        cell.paragraphs[0].alignment = WD_ALIGN_PARAGRAPH.CENTER
        for run in cell.paragraphs[0].runs:
            run.font.bold = True
            run.font.size = Pt(10)
            run.font.color.rgb = COLORS["primary"]

        # Background (simulado con shading)
        self._set_cell_shading(cell, "ECF0F1")

    def _style_data_cell(self, cell, center: bool = False, bold: bool = False, font_size: int = 10):
        """Estilizar celda de datos."""
        if center:
            cell.paragraphs[0].alignment = WD_ALIGN_PARAGRAPH.CENTER
        for run in cell.paragraphs[0].runs:
            run.font.size = Pt(font_size)
            if bold:
                run.font.bold = True

    def _set_cell_shading(self, cell, color_hex: str):
        """Establecer color de fondo de celda."""
        shading = OxmlElement("w:shd")
        shading.set(qn("w:fill"), color_hex)
        cell._tc.get_or_add_tcPr().append(shading)

    # =========================================================================
    # Helpers de formateo
    # =========================================================================

    def _format_currency(self, value: float) -> str:
        """Formatear valor como moneda (formato español)."""
        if value >= 1000:
            return f"{value/1000:.1f}K EUR"
        # Formato español: punto como separador de miles
        return f"{value:,.0f} EUR".replace(",", ".")

    def _format_period(self, start: date, end: date) -> str:
        """Formatear período para mostrar."""
        months = [
            "Enero", "Febrero", "Marzo", "Abril", "Mayo", "Junio",
            "Julio", "Agosto", "Septiembre", "Octubre", "Noviembre", "Diciembre"
        ]

        if start.month == end.month and start.year == end.year:
            return f"{months[start.month - 1]} {start.year}"
        else:
            return f"{start.day}/{start.month}/{start.year} - {end.day}/{end.month}/{end.year}"


# Instancia singleton
word_action_plan_generator = WordActionPlanGenerator()
